<?php
if (!defined('ABSPATH')) {
    exit;
}

class Corelentra_License
{

    private static $instance = null;
    private $api_url = 'https://corelentra.com/api/license'; // Placeholder
    private $option_key = 'corelentra_license_key';
    private $status_key = 'corelentra_license_status';

    public static function instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private $check_interval = 86400; // 24 hours

    private function __construct()
    {
        add_action('admin_menu', array($this, 'register_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_init', array($this, 'daily_license_check'));
        add_action('init', array($this, 'register_shortcodes'));
    }

    public function register_shortcodes()
    {
        add_shortcode('corelentra_pro_pricing', array($this, 'render_pricing_shortcode'));
    }

    public function render_pricing_shortcode($atts)
    {
        $pricing = get_transient('corelentra_pricing_cache');

        if (false === $pricing) {
            $response = wp_remote_get('https://corelentra.com/api/pricing', array('timeout' => 5));
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $data = json_decode(wp_remote_retrieve_body($response), true);
                if (!empty($data['price'])) {
                    $pricing = $data['price'];
                    set_transient('corelentra_pricing_cache', $pricing, 12 * HOUR_IN_SECONDS);
                }
            }
        }

        if (!$pricing) {
            $pricing = '$99 / year';
        }

        return '<div class="corelentra-pricing-box" style="border:1px solid #ddd; padding:20px; border-radius:5px; text-align:center; background:#f9f9f9;">
            <h3>Upgrade to Pro</h3>
            <p class="price" style="font-size:24px; font-weight:bold; color:#2271b1;">' . esc_html($pricing) . '</p>
            <ul style="text-align:left; margin:15px auto; display:inline-block;">
                <li>✓ Vendor Management</li>
                <li>✓ Email Contractors Logic</li>
                <li>✓ Google Calendar Sync</li>
                <li>✓ Custom Email Templates</li>
                <li>✓ PDF Pricelist Downloads</li>
            </ul>
            <p><a href="https://corelentra.com/pricing" target="_blank" class="button button-primary">Get Pro Now</a></p>
        </div>';
    }

    public function daily_license_check()
    {
        // Only run for admins to avoid frontend overhead
        if (!current_user_can('manage_options')) {
            return;
        }

        $last_check = get_option('corelentra_license_last_check', 0);
        if ((time() - $last_check) > $this->check_interval) {
            $this->validate_license_silently();
        }
    }

    public function register_menu()
    {
        add_submenu_page(
            'corelentra-main',
            __('License', 'corelentra'),
            __('License', 'corelentra'),
            'manage_options',
            'corelentra-license',
            array($this, 'render_page')
        );
    }

    public function register_settings()
    {
        register_setting('corelentra_license_group', $this->option_key, array('sanitize_callback' => array($this, 'sanitize_license_key')));
    }

    public function sanitize_license_key($new_value)
    {
        $old_value = get_option($this->option_key);
        if ($old_value !== $new_value) {
            update_option($this->status_key, 'inactive');
        }
        return sanitize_text_field($new_value);
    }

    public function render_page()
    {
        $key = get_option($this->option_key, '');
        $status = get_option($this->status_key, 'inactive');
        $last_check = get_option('corelentra_license_last_check', 0);
        $active_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'activation';
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e('Corelentra License', 'corelentra'); ?></h1>
            <a href="https://corelentra.com/pricing" target="_blank"
                class="page-title-action"><?php esc_html_e('Buy License', 'corelentra'); ?></a>
            <hr class="wp-header-end">

            <nav class="nav-tab-wrapper">
                <a href="?page=corelentra-license&tab=activation"
                    class="nav-tab <?php echo $active_tab === 'activation' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e('Activation', 'corelentra'); ?></a>
                <a href="?page=corelentra-license&tab=help"
                    class="nav-tab <?php echo $active_tab === 'help' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e('Help & Setup', 'corelentra'); ?></a>
            </nav>

            <div class="tab-content" style="background:#fff; border:1px solid #ccd0d4; border-top:none; padding:20px;">
                <?php if ($active_tab === 'help'): ?>
                    <h2><?php esc_html_e('How to Setup Corelentra', 'corelentra'); ?></h2>
                    <ol>
                        <li><strong><?php esc_html_e('Purchase a License:', 'corelentra'); ?></strong>
                            <?php esc_html_e('Visit Corelentra.com and purchase the Corelentra Pro Suite.', 'corelentra'); ?></li>
                        <li><strong><?php esc_html_e('Get your Key:', 'corelentra'); ?></strong>
                            <?php esc_html_e('Log in to your account dashboard and copy the license key for your purchase.', 'corelentra'); ?>
                        </li>
                        <li><strong><?php esc_html_e('Activate:', 'corelentra'); ?></strong>
                            <?php esc_html_e('Paste the key in the "Activation" tab and click "Save & Validate".', 'corelentra'); ?>
                        </li>
                    </ol>
                    <p><em><?php esc_html_e('Note: Pro features like Google Sync, Vendor Management, and PDF Tools will remain locked until a valid key is activated.', 'corelentra'); ?></em>
                    </p>

                    <h3><?php esc_html_e('Troubleshooting', 'corelentra'); ?></h3>
                    <p><?php esc_html_e('If your key is rejected:', 'corelentra'); ?></p>
                    <ul>
                        <li><?php esc_html_e('Ensure you have copied the full key without spaces.', 'corelentra'); ?></li>
                        <li><?php esc_html_e('Check if your license has expired.', 'corelentra'); ?></li>
                    </ul>

                <?php else: ?>
                    <!-- Activation Tab -->
                    <div>
                        <?php
                        // Handle Save & Validate logic here
                        if (isset($_POST['corelentra_action']) && $_POST['corelentra_action'] === 'save_validate') {
                            check_admin_referer('corelentra_license_action');

                            $new_key = sanitize_text_field($_POST[$this->option_key]);
                            update_option($this->option_key, $new_key);

                            // Triggers validation/update logic
                            $this->activate_license($new_key);

                            // Refresh local vars
                            $key = get_option($this->option_key, '');
                            $status = get_option($this->status_key, 'inactive');
                            $last_check = get_option('corelentra_license_last_check', 0);
                        }
                        ?>

                        <form method="post">
                            <?php wp_nonce_field('corelentra_license_action'); ?>
                            <input type="hidden" name="corelentra_action" value="save_validate">

                            <table class="form-table">
                                <tr valign="top">
                                    <th scope="row"><?php esc_html_e('License Key', 'corelentra'); ?></th>
                                    <td><input type="text" name="<?php echo esc_attr($this->option_key); ?>"
                                            value="<?php echo esc_attr($key); ?>" class="regular-text" /></td>
                                </tr>
                                <tr valign="top">
                                    <th scope="row"><?php esc_html_e('Status', 'corelentra'); ?></th>
                                    <td>
                                        <?php if ($status === 'valid'): ?>
                                            <span
                                                style="color:green;font-weight:bold;"><?php esc_html_e('Active Pro', 'corelentra'); ?></span>
                                        <?php else: ?>
                                            <span
                                                style="color:red;"><?php esc_html_e('Inactive / Free Version', 'corelentra'); ?></span>
                                        <?php endif; ?>
                                        <?php if ($last_check): ?>
                                            <p class="description">
                                                <?php printf(__('Last checked: %s', 'corelentra'), date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_check)); ?>
                                            </p>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            </table>

                            <p class="submit">
                                <button type="submit"
                                    class="button button-primary button-large"><?php esc_html_e('Save & Validate License', 'corelentra'); ?></button>
                            </p>
                        </form>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }



    public function activate_license($key)
    {
        $result = $this->perform_validation($key);

        if ($result['success']) {
            echo '<div class="notice notice-success"><p>' . esc_html__('License activated successfully! Pro features unlocked.', 'corelentra') . '</p></div>';
        } else {
            echo '<div class="notice notice-error"><p>' . esc_html($result['message']) . '</p></div>';
        }
    }

    private function validate_license_silently()
    {
        $key = get_option($this->option_key, '');
        $this->perform_validation($key);
    }

    private function perform_validation($key)
    {
        // Always update check time
        update_option('corelentra_license_last_check', time());

        if (empty($key)) {
            update_option($this->status_key, 'inactive');
            return array('success' => false, 'message' => __('Please enter a license key.', 'corelentra'));
        }

        /* 
           MOCK VALIDATION FOR TESTING
           Since the API endpoint is a placeholder, any non-empty key is treated as VALID.
        */
        update_option($this->status_key, 'valid');
        return array('success' => true, 'message' => 'License activated validly (Dev Mode).');

        // Original Remote Check Logic (Disabled for testing)
        /*
        $response = wp_remote_post($this->api_url, array(
            'timeout' => 15,
            'body' => array(
                'action' => 'activate_license',
                'license_key' => $key,
                'product_id' => 'corelentra_pro', // Added for server validation
                'site_url' => home_url(),
            ),
        ));

        if (is_wp_error($response)) {
            return array('success' => false, 'message' => __('Connection failed: ' . $response->get_error_message(), 'corelentra'));
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($code === 200 && isset($data['success']) && $data['success'] === true) {
            update_option($this->status_key, 'valid');
            return array('success' => true, 'message' => '');
        } else {
            update_option($this->status_key, 'inactive');
            $error_msg = isset($data['data']['message']) ? $data['data']['message'] : __('Invalid license key or server error.', 'corelentra');
            return array('success' => false, 'message' => $error_msg);
        }
        */
    }

    public static function is_pro()
    {
        return get_option('corelentra_license_status', 'inactive') === 'valid';
    }
}
