<?php
/*
Module Name: DFS Pricelist (Component)
Description: This is a in Build CRM Plugin for Service Based Business including a emailer for emailing Customers various Email Templates.
Version: 2.0
Author: Kurt Lynch
*/

# Plugin Details:
# This plugin includes the following features:
# - Display WooCommerce products by category in a catalog format
# - Show product images, prices, and compared prices
# - Include product descriptions and short descriptions
# - Customize category and subcategory background colors
# - Filter which categories are included in the pricelist
# - Generate a downloadable PDF version of the catalog
# - Centered black "Download PDF" button on the Settings page
# - Full Settings panel with shortcode references:
#     • [dfs_pricelist] – Display the product catalog in HTML
#     • [dfs_pricelist_pdf_button] – PDF download button
#
# ⚠️ Note: Exclude Product Bundles feature is currently not working properly.


add_action('admin_post_save_dfs_pricelist_settings', function () {
    if (!current_user_can('manage_options'))
        wp_die('Unauthorized');

    check_admin_referer('dfs_pricelist_settings_verify');

    $fields = [
        'company',
        'phone',
        'website',
        'include_compared',
        'include_description',
        'include_short_description',
        'exclude_bundled',
        'cat_bg',
        'subcat_bg',
        'categories'
    ];

    $cleaned = [];

    foreach ($fields as $field) {
        if ($field === 'categories') {
            $cleaned[$field] = array_map('intval', $_POST['dfs_pricelist_settings'][$field] ?? []);
        } else {
            $cleaned[$field] = sanitize_text_field($_POST['dfs_pricelist_settings'][$field] ?? '');
        }
    }

    update_option('dfs_pricelist_settings', $cleaned);
    wp_redirect(admin_url('admin.php?page=dfs_settings&status=1'));
    exit;
});



add_action('admin_init', function () {
    register_setting('dfs_settings', 'dfs_pricelist_settings');
});


/*
Module Name: DFS Pricelist
Description: Display and export WooCommerce product prices by category.
Version: 2.3.4
Author: Kurt Lynch
*/




function dfs_pricelist_shortcode()
{
    if (!function_exists('wc_get_products'))
        return '<p>WooCommerce not available.</p>';
    $options = get_option('dfs_pricelist_settings');
    $selected = $options['categories'] ?? [];
    $cat_bg = esc_attr($options['cat_bg'] ?? '#f0f0f0');

    ob_start();
    foreach ($selected as $cat_id) {
        $term = get_term($cat_id);
        if (!$term || is_wp_error($term))
            continue;

        echo "<h3 style='background-color:{$cat_bg};padding:8px;font-size:16px;'>{$term->name}</h3>";
        echo "<table class='widefat striped' style='width:100%;margin-bottom:20px;'>";
        echo "<thead><tr>"
            . "<th style='width:60px;'>Image</th>"
            . "<th>Name</th>"
            . "<th style='width:120px;'>Price</th>"
            . "<th style='width:140px;'>Compared Price</th>"
            . "<th style='width:200px;white-space:nowrap;'>Actions</th>"
            . "</tr></thead><tbody>";

        $products = wc_get_products([
            'type' => 'simple',
            'status' => 'publish',
            'limit' => -1,
            'category' => [$term->slug]

        ]);

        foreach ($products as $product) {
            if (method_exists($product, 'get_type') && $product->get_type() === 'bundle')
                continue;
            if (!empty($options['exclude_bundled'])) {
                $type = is_object($product) && method_exists($product, 'get_type') ? $product->get_type() : '';
                if ($type === 'bundle')
                    continue;
            }

            $id = $product->get_id();
            $img = preg_replace(
                '/<img /',
                '<img style="width:50px;height:auto;border-radius:4px;margin-right:8px;" ',
                $product->get_image('thumbnail'),
                1
            );
            $price = wc_price($product->get_price());
            $compared = !empty($options['include_compared']) ? wc_price($product->get_regular_price()) : '';
            $view_link = get_permalink($id);

            if (is_admin()) {
                $edit_link = get_edit_post_link($id);
                $actions = "<a href='{$edit_link}'>Edit</a> | <a href='{$view_link}' target='_blank'>View</a>";
            } else {
                $actions = "<a href='{$view_link}' target='_blank'>View</a>";
            }

            echo "<tr>"
                . "<td>{$img}</td>"
                . "<td>" . esc_html($product->get_name()) . "</td>"
                . "<td>{$price}</td>"
                . "<td>{$compared}</td>"
                . "<td>{$actions}</td>"
                . "</tr>";
        }

        echo "</tbody></table>";
    }
    return ob_get_clean();
}
add_shortcode('dfs_pricelist', 'dfs_pricelist_shortcode');





add_action('init', function () {
    if (isset($_POST['dfs_download_pdf'])) {
        $tcpdf_path = dirname(__FILE__) . '/tcpdf/tcpdf.php';
        if (file_exists($tcpdf_path)) {
            require_once $tcpdf_path;
        }
        if (!class_exists('TCPDF')) {
            echo 'PDF generation not available: TCPDF library could not be loaded.';
            exit;
        }
        $options = get_option('dfs_pricelist_settings');
        $selected = $options['categories'] ?? [];
        $pdf = new TCPDF();
        $pdf->AddPage();

        $pdf->SetFont('helvetica', '', 10);
        $html = '<h1 style="text-align:center;">' . esc_html(($options['company'] ?? 'Price List') . ' - ' . ($options['phone'] ?? '')) . '</h1>';
        $html .= '<h3 style="text-align:center;">' . esc_html($options['website'] ?? '') . '</h3>';
        foreach ($selected as $cat_id) {
            $term = get_term($cat_id);
            if (!$term || is_wp_error($term))
                continue;
            $html .= '<h2>' . esc_html($term->name) . '</h2>';
            $html .= '<table border="1" cellpadding="4" cellspacing="0" width="100%" style="border-collapse:collapse;">';
            $html .= '<thead><tr>'
                . '<th>Name</th>'
                . '<th width="120px">Price</th>'
                . '<th width="140px">Compared Price</th>'
                . '<th>View</th>'
                . '</tr></thead><tbody>';
            $products = wc_get_products([
                'type' => 'simple',
                'status' => 'publish',
                'limit' => -1,
                'category' => [$term->slug]

            ]);
            foreach ($products as $product) {
                if (method_exists($product, 'get_type') && $product->get_type() === 'bundle')
                    continue;
                $html .= '<tr>'
                    . '<td>' . esc_html($product->get_name()) . '</td>'
                    . '<td width="120px">' . wc_price($product->get_price()) . '</td>'
                    . '<td width="140px">' . (!empty($options['include_compared']) ? wc_price($product->get_regular_price()) : '') . '</td>'
                    . '<td><a href="' . get_permalink($product->get_id()) . '" target="_blank">View</a></td>'
                    . '</tr>';
            }
            $html .= '</tbody></table>';
        }
        $pdf->writeHTML($html, true, false, true, false, '');
        $pdf->Output('dfs-pricelist.pdf', 'D');
        exit;
    }
});








function dfs_pricelist_settings_page()
{
    if (function_exists('dfs_pricelist_settings')) {
        dfs_pricelist_settings();
    } else {
        echo '<div class="wrap"><h1>DFS Pricelist Settings</h1><p>Settings page not found.</p></div>';
    }
}

function dfs_pricelist_admin_page()
{
    echo '<div class="wrap"><h1>HTML Pricelist View</h1>';
    echo do_shortcode('[dfs_pricelist]');
    echo '</div>';
}







function dfs_pricelist_settings()
{
    $options = get_option('dfs_pricelist_settings');
    $categories = get_terms(array('taxonomy' => 'product_cat', 'hide_empty' => false));

    echo <<<HTML
    <div class="wrap">
        <h1>DFS Pricelist</h1>

HTML;

    echo '<form method="post"><p style="text-align:center;">';

    if (class_exists('Corelentra_License') && Corelentra_License::is_pro()) {
        echo '<button type="submit" name="dfs_download_pdf" style="background:#000;color:#fff;padding:10px 20px;border:none;border-radius:5px;">Download PDF</button>';
    } else {
        echo '<button type="button" disabled style="background:#ccc;color:#666;padding:10px 20px;border:none;border-radius:5px;cursor:not-allowed;">Download PDF (Pro Only)</button>';
    }

    echo '</p></form>';

    echo <<<HTML
    

        <h2>Available Shortcodes</h2>
        <ul>
            <li><code>[dfs_pricelist]</code> – Display full WooCommerce price list by category</li>
            <li><code>[dfs_pricelist_pdf_button]</code> – Display a download button for PDF version</li>
        </ul>

        <form method="post" action="admin-post.php">
HTML;

    wp_nonce_field("dfs_pricelist_settings_verify");
    echo '<input type="hidden" name="action" value="save_dfs_pricelist_settings" />';

    echo '<table class="form-table">';
    echo '<tr><th scope="row">Company Name</th><td><input type="text" name="dfs_pricelist_settings[company]" value="' . esc_attr($options['company'] ?? '') . '" class="regular-text" /></td></tr>';
    echo '<tr><th scope="row">Phone Number</th><td><input type="text" name="dfs_pricelist_settings[phone]" value="' . esc_attr($options['phone'] ?? '') . '" class="regular-text" /></td></tr>';
    echo '<tr><th scope="row">Website URL</th><td><input type="text" name="dfs_pricelist_settings[website]" value="' . esc_attr($options['website'] ?? '') . '" class="regular-text" /></td></tr>';
    echo '<tr><th scope="row">Include Compared Price</th><td><label><input type="checkbox" name="dfs_pricelist_settings[include_compared]" value="1" ' . checked($options['include_compared'] ?? '', 1, false) . '> Yes</label></td></tr>';
    echo '<tr><th scope="row">Include Product Description</th><td><label><input type="checkbox" name="dfs_pricelist_settings[include_description]" value="1" ' . checked($options['include_description'] ?? '', 1, false) . '> Yes</label></td></tr>';
    echo '<tr><th scope="row">Include Short Description</th><td><label><input type="checkbox" name="dfs_pricelist_settings[include_short_description]" value="1" ' . checked($options['include_short_description'] ?? '', 1, false) . '> Yes</label></td></tr>';
    echo '<tr><th scope="row">Exclude Bundled Products</th><td><label><input type="checkbox" name="dfs_pricelist_settings[exclude_bundled]" value="1" ' . checked($options['exclude_bundled'] ?? '', 1, false) . '> Yes</label></td></tr>';
    echo '<tr><th scope="row">Category Background Color</th><td><input type="color" name="dfs_pricelist_settings[cat_bg]" value="' . esc_attr($options['cat_bg'] ?? '#f0f0f0') . '" /></td></tr>';
    echo '<tr><th scope="row">Subcategory Background Color</th><td><input type="color" name="dfs_pricelist_settings[subcat_bg]" value="' . esc_attr($options['subcat_bg'] ?? '#ffffff') . '" /></td></tr>';
    echo '<tr><th scope="row">Select Categories to Include</th><td>';

    foreach ($categories as $cat) {
        $checked = isset($options['categories']) && in_array($cat->term_id, (array) $options['categories']) ? 'checked' : '';
        echo '<label><input type="checkbox" name="dfs_pricelist_settings[categories][]" value="' . esc_attr($cat->term_id) . '" ' . $checked . '> ' . esc_html($cat->name) . '</label><br>';
    }

    echo '</td></tr>';
    echo '</table>';

    submit_button();
    echo '</form></div>';
}


function dfs_pricelist_pdf_button_shortcode()
{
    if (!class_exists('Corelentra_License') || !Corelentra_License::is_pro()) {
        return '<p><button type="button" disabled style="background:#ccc;color:#666;padding:10px 20px;border:none;border-radius:5px;cursor:not-allowed;">Download PDF (Pro Only)</button></p>';
    }
    return '<form method="post"><p style="text-align:center;"><button type="submit" name="dfs_download_pdf" style="background:#000;color:#fff;padding:10px 20px;border:none;border-radius:5px;">Download PDF</button></p></form>';
}
add_shortcode('dfs_pricelist_pdf_button', 'dfs_pricelist_pdf_button_shortcode');


// Render Edit Products page

// Handle bulk update
add_action('admin_init', function () {
    if (isset($_POST['products']) && current_user_can('manage_woocommerce')) {
        foreach ($_POST['products'] as $id => $fields) {
            if (isset($fields['name']))
                wp_update_post(['ID' => $id, 'post_title' => sanitize_text_field($fields['name'])]);
            if (isset($fields['sku']))
                update_post_meta($id, '_sku', sanitize_text_field($fields['sku']));
            if (isset($fields['regular_price']))
                update_post_meta($id, '_regular_price', wc_format_decimal($fields['regular_price']));
            if (isset($fields['price']))
                update_post_meta($id, '_price', wc_format_decimal($fields['price']));
            if (isset($fields['cogs']))
                update_post_meta($id, '_cogs_price', wc_format_decimal($fields['cogs']));
        }
        wp_redirect(admin_url('admin.php?page=dfs_edit_products&updated=1'));
        exit;
    }
});

// Add Fence Contractors menu

// Create contractor table on plugin activation

// Render the contractor list page
function dfs_render_fence_contractors()
{
    // Pro Restriction: Lock entire Contractors page if not Pro
    if (class_exists('Corelentra_License') && !Corelentra_License::is_pro()) {
        echo '<div class="wrap"><h1>Fence Contractors</h1>';
        echo '<div style="background:#fff;border:1px solid #ccd0d4;padding:20px;text-align:center;max-width:600px;margin:20px 0;">';
        echo '<h2 style="color:#d63638;">Pro Feature Locked</h2>';
        echo '<p style="font-size:16px;">Vendor and Contractor management, including the "Email Selected" workflow, is available only in <strong>Corelentra Pro</strong>.</p>';
        echo '<p><a href="' . admin_url('admin.php?page=corelentra-license') . '" class="button button-primary button-large">Activate License</a></p>';
        echo '</div></div>';
        return;
    }

    // Email Selected Modal (with TinyMCE)
    echo '<div id="dfs-email-modal" style="display:none;position:fixed;top:50%;left:50%;transform:translate(-50%, -50%);width:65%;height:65%;background:#fff;z-index:9999;overflow:auto;padding:20px;border:1px solid #ccc;box-shadow:0 0 10px rgba(0,0,0,.5);">';
    echo '<button id="dfs-email-close" style="float:right;">&times;</button>';
    echo '<h2>Email Contractors</h2>';
    echo '<select id="dfs-template-select"><option value="">-- Select Template --</option></select>';
    echo '<button id="dfs-save-template" class="button">Save Template</button>';
    echo '<button id="dfs-edit-template" class="button">Save Edit Template</button>';
    echo '<button id="dfs-delete-template" class="button">Delete Template</button>';
    echo '<p><input type="text" id="dfs-email-subject" placeholder="Subject" style="width:100%;" /></p>';
    echo '<p>';
    wp_editor(
        '',
        'dfs-email-body',
        array(
            'textarea_name' => 'dfs-email-body',
            'editor_height' => 300,
            'tinymce' => true,
            'quicktags' => true
        )
    );
    echo '</p>';
    echo '<button id="dfs-send-emails" class="button button-primary">Send</button>';
    echo '</div>';
    // Progress Window
    echo '<div id="dfs-email-progress" style="display:none;position:fixed;top:30%;left:40%;width:25%;height:40%;background:#f9f9f9;z-index:9999;overflow:auto;padding:20px;border:1px solid #ccc;box-shadow:0 0 10px rgba(0,0,0,.5);">'
        . '<button id="dfs-progress-close" style="float:right;">&times;</button>'
        . '<h2>Email Progress</h2>'
        . '<ul id="dfs-progress-list" style="list-style:none;padding:0;margin:0;max-height:300px;overflow-y:auto;"></ul>'
        . '</div>';

    // Handle CSV/TXT file import

    if (isset($_POST['import_submit']) && !empty($_FILES['import_file']['tmp_name'])) {

        $raw = file_get_contents($_FILES['import_file']['tmp_name']);

        $lines = preg_split('/\r\n|\n/', trim($raw));

        global $wpdb;

        $table = $wpdb->prefix . 'dfs_contractors';

        foreach ($lines as $line) {

            $delim = strpos($line, ':') !== false ? ':' : ',';

            $parts = array_map('trim', explode($delim, $line));

            if (count($parts) >= 8) {

                list($vendor, $contact, $address, $city, $state, $zip, $phone, $email) = $parts;

                $wpdb->insert($table, [

                    'contractor_name' => sanitize_text_field($vendor),

                    'contact_name' => sanitize_text_field($contact),

                    'address' => sanitize_text_field($address),

                    'city' => sanitize_text_field($city),

                    'state' => sanitize_text_field($state),

                    'zip' => sanitize_text_field($zip),

                    'phone' => sanitize_text_field($phone),

                    'email' => sanitize_email($email),

                ]);

            }

        }

        wp_redirect(admin_url('admin.php?page=dfs_fence_contractors&imported=' . count($lines)));

        exit;

    }

    if (isset($_GET['imported'])) {

        echo '<div class="notice notice-success is-dismissible"><p>Imported ' . intval($_GET['imported']) . ' contractors.</p></div>';

    }

    echo '<div class="wrap"><h1>Fence Contractors</h1>';
    echo '<button id="dfs_add_contractor" class="button button-primary">Add New Contractor</button>';
    echo '<form method="post" enctype="multipart/form-data" style="display:inline-block;margin-left:10px;">';

    echo '<input type="file" name="import_file" accept=".csv,.txt" />';

    echo '<button type="submit" name="import_submit" class="button">Import File</button>';

    echo '</form>';

    echo '<button id="dfs_email_selected" class="button">Email Selected</button>';

    echo '<br><br><label for="dfs_per_page">Show per page: </label>';
    echo '<select id="dfs_per_page"><option>100</option><option>200</option><option>300</option><option>600</option><option>900</option></select>';

    echo '<form method="post"><table class="widefat fixed striped"><thead>
        <tr>
            <th><input type="checkbox" id="dfs_select_all" /></th>
                <th>Actions</th>
            <th>Contractor Name</th>
            <th>Contact Name</th>
            <th>Address</th>
            <th>City</th>
            <th>State</th>
            <th>Zip</th>
            <th>Phone</th>
            <th>Email</th>
            <th>Opt-Out</th>
        </tr>
    </thead><tbody>';

    global $wpdb;
    $table_name = $wpdb->prefix . 'dfs_contractors';
    $contractors = $wpdb->get_results("SELECT * FROM $table_name ORDER BY id DESC");

    foreach ($contractors as $c) {
        echo '<tr>'
            . '<td><input type="checkbox" class="dfs_check_row" name="contractors[]" value="' . esc_attr($c->id) . '" /></td>'
            . '<td><a href="#" class="dfs_edit" data-id="' . esc_attr($c->id) . '">Edit</a> | <a href="#" class="dfs_delete" data-id="' . esc_attr($c->id) . '">Delete</a></td>'
            . '<td>' . esc_html($c->contractor_name) . '</td>'
            . '<td>' . esc_html($c->contact_name) . '</td>'
            . '<td>' . esc_html($c->address) . '</td>'
            . '<td>' . esc_html($c->city) . '</td>'
            . '<td>' . esc_html($c->state) . '</td>'
            . '<td>' . esc_html($c->zip) . '</td>'
            . '<td>' . esc_html($c->phone) . '</td>'
            . '<td>' . esc_html($c->email) . '</td>'
            . '<td><input type="checkbox" disabled ' . ($c->opt_out ? 'checked' : '') . ' /></td>'
            . '</tr>';

    }

    echo '</tbody></table></form></div>';
}

// Include modal HTML and JS for Add/Edit/Delete
add_action('admin_footer', function () {
    if (!isset($_GET['page']) || $_GET['page'] !== 'dfs_fence_contractors')
        return;

    // AJAX: Get email template content
    add_action('wp_ajax_dfs_get_email_template', 'dfs_get_email_template');
    function dfs_get_email_template()
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        $id = isset($_POST['template_id']) ? intval($_POST['template_id']) : 0;
        if (!$id) {
            wp_send_json_error('Invalid template ID');
        }
        global $wpdb;
        $table = $wpdb->prefix . 'dfs_email_templates';
        $row = $wpdb->get_row($wpdb->prepare("SELECT subject, body FROM {$table} WHERE id = %d", $id));
        if ($row) {
            wp_send_json_success(array('subject' => $row->subject, 'body' => $row->body));
        } else {
            wp_send_json_error('Template not found');
        }
    }


    // Remove duplicate parent submenu link
    add_action('admin_menu', function () {
        remove_submenu_page('dfs_settings', 'dfs_settings');
    }, 999);

    ?>
    <div id="dfs_contractor_modal"
        style="display:none;position:fixed;top:10%;left:20%;width:60%;background:#fff;padding:20px;border:1px solid #ccc;z-index:9999;">
        <h2 id="dfs_modal_title">Add Contractor</h2>
        <form id="dfs_contractor_form">
            <input type="hidden" name="id" id="contractor_id">
            <p><label>Contractor Name:</label><br><input type="text" name="contractor_name" id="contractor_name"
                    style="width:100%;"></p>
            <p><label>Contact Name:</label><br><input type="text" name="contact_name" id="contact_name" style="width:100%;">
            </p>
            <p><label>Address:</label><br><input type="text" name="address" id="address" style="width:100%;"></p>
            <p><label>City:</label><br><input type="text" name="city" id="city" style="width:100%;"></p>
            <p><label>State:</label><br><input type="text" name="state" id="state" style="width:100%;"></p>
            <p><label>Zip:</label><br><input type="text" name="zip" id="zip" style="width:100%;"></p>
            <p><label>Phone:</label><br><input type="text" name="phone" id="phone" style="width:100%;"></p>
            <p><label>Email:</label><br><input type="email" name="email" id="email" style="width:100%;"></p>
            <p><label><input type="checkbox" name="opt_out" id="opt_out"> Do not email</label>
            </p>
            <p>
                <button type="submit" class="button button-primary">Save</button>
                <button type="button" id="dfs_cancel_modal" class="button">Cancel</button>
            </p>
        </form>
    </div>
    <script>
        jQuery(document).ready(function ($) {
            $('#dfs_add_contractor').on('click', function () {
                $('#dfs_contractor_form')[0].reset();
                $('#contractor_id').val('');
                $('#dfs_modal_title').text('Add Contractor');
                $('#dfs_contractor_modal').show();
            });

            $('.dfs_edit').on('click', function (e) {
                e.preventDefault();
                var id = $(this).data('id');
                $.post(ajaxurl, { action: 'dfs_get_contractor', id: id }, function (data) {
                    if (data.success) {
                        var c = data.data;
                        $('#contractor_id').val(c.id);
                        $('#contractor_name').val(c.contractor_name);
                        $('#contact_name').val(c.contact_name);
                        $('#address').val(c.address);
                        $('#city').val(c.city);
                        $('#state').val(c.state);
                        $('#zip').val(c.zip);
                        $('#phone').val(c.phone);
                        $('#email').val(c.email);
                        $('#opt_out').prop('checked', c.opt_out == 1);
                        $('#dfs_modal_title').text('Edit Contractor');
                        $('#dfs_contractor_modal').show();
                    }
                });
            });

            $('#dfs_cancel_modal').on('click', function () {
                $('#dfs_contractor_modal').hide();
            });

            $('#dfs_contractor_form').on('submit', function (e) {
                e.preventDefault();
                var formData = $(this).serialize();
                $.post(ajaxurl, formData + '&action=dfs_save_contractor', function (response) {
                    if (response.success) location.reload();
                });
            });

            $('.dfs_delete').on('click', function (e) {
                e.preventDefault();
                if (confirm('Are you sure?')) {
                    var id = $(this).data('id');
                    $.post(ajaxurl, { action: 'dfs_delete_contractor', id: id }, function (resp) {
                        if (resp.success) location.reload();
                    });
                }
            });
        });
    </script>
    <?php
});

// AJAX: get contractor
add_action('wp_ajax_dfs_get_contractor', function () {
    global $wpdb;
    $id = intval($_POST['id']);
    $table = $wpdb->prefix . 'dfs_contractors';
    $contractor = $wpdb->get_row("SELECT * FROM $table WHERE id = $id", ARRAY_A);
    wp_send_json_success($contractor);
});

// AJAX: save contractor (insert/update)
add_action('wp_ajax_dfs_save_contractor', function () {
    global $wpdb;
    $table = $wpdb->prefix . 'dfs_contractors';
    $data = [
        'contractor_name' => sanitize_text_field($_POST['contractor_name']),
        'contact_name' => sanitize_text_field($_POST['contact_name']),
        'address' => sanitize_text_field($_POST['address']),
        'city' => sanitize_text_field($_POST['city']),
        'state' => sanitize_text_field($_POST['state']),
        'zip' => sanitize_text_field($_POST['zip']),
        'phone' => sanitize_text_field($_POST['phone']),
        'email' => sanitize_email($_POST['email']),
        'opt_out' => isset($_POST['opt_out']) ? 1 : 0,
    ];
    if (!empty($_POST['id'])) {
        $wpdb->update($table, $data, ['id' => intval($_POST['id'])]);
    } else {
        $wpdb->insert($table, $data);
    }
    wp_send_json_success();
});

// AJAX: delete contractor
add_action('wp_ajax_dfs_delete_contractor', function () {
    global $wpdb;
    $id = intval($_POST['id']);
    $wpdb->delete($wpdb->prefix . 'dfs_contractors', ['id' => $id]);
    wp_send_json_success();
});

// Include import modal UI
add_action('admin_footer', function () {
    if (!isset($_GET['page']) || $_GET['page'] !== 'dfs_fence_contractors')
        return;

    // Remove duplicate parent submenu link
    add_action('admin_menu', function () {
        remove_submenu_page('dfs_settings', 'dfs_settings');
    }, 999);

    ?>
    <div id="dfs_import_modal"
        style="display:none;position:fixed;top:10%;left:20%;width:60%;background:#fff;padding:20px;border:1px solid #ccc;z-index:9999;">
        <h2>Import Contractors</h2>
        <form id="dfs_import_form">
            <p>Paste CSV or semicolon-delimited text (one contractor per line):</p>
            <textarea id="import_data" name="import_data" rows="10" style="width:100%;"></textarea>
            <p><button type="submit" class="button button-primary">Import</button>
                <button type="button" id="dfs_cancel_import" class="button">Cancel</button>
            </p>
        </form>
    </div>
    <script>
        jQuery(document).ready(function ($) {
            $('#dfs_import_contractors').on('click', function () {
                $('#dfs_import_modal').show();
            });
            $('#dfs_cancel_import').on('click', function () {
                $('#dfs_import_modal').hide();
            });
            $('#dfs_import_form').on('submit', function (e) {
                e.preventDefault();
                var data = $('#import_data').val();
                $.post(ajaxurl, { action: 'dfs_import_contractors', import_data: data }, function (resp) {
                    if (resp.success) {
                        alert('Import complete: ' + resp.data.imported + ' records added.');
                        location.reload();
                    } else {
                        alert('Import failed.');
                    }
                });
            });
        });
    </script>
    <?php
});

// Handle import
add_action('wp_ajax_dfs_import_contractors', function () {
    global $wpdb;
    $lines = explode("\n", stripslashes($_POST['import_data']));
    $imported = 0;
    $table = $wpdb->prefix . 'dfs_contractors';

    foreach ($lines as $line) {
        $line = trim($line);
        if (!$line)
            continue;

        $delimiter = strpos($line, ';') !== false ? ';' : ',';
        $parts = array_map('trim', explode($delimiter, $line));
        if (count($parts) < 8)
            continue;

        list($contractor_name, $contact_name, $address, $city, $state, $zip, $phone, $email) = array_pad($parts, 8, '');
        $opt_out = isset($parts[8]) && strtolower($parts[8]) === 'yes' ? 1 : 0;

        $wpdb->insert($table, [
            'contractor_name' => $contractor_name,
            'contact_name' => $contact_name,
            'address' => $address,
            'city' => $city,
            'state' => $state,
            'zip' => $zip,
            'phone' => $phone,
            'email' => $email,
            'opt_out' => $opt_out,
        ]);
        $imported++;
    }

    wp_send_json_success(['imported' => $imported]);
});

// Include enhanced import modal UI
add_action('admin_footer', function () {
    if (!isset($_GET['page']) || $_GET['page'] !== 'dfs_fence_contractors')
        return;

    // Remove duplicate parent submenu link
    add_action('admin_menu', function () {
        remove_submenu_page('dfs_settings', 'dfs_settings');
    }, 999);

    ?>
    <div id="dfs_import_modal"
        style="display:none;position:fixed;top:5%;left:15%;width:70%;background:#fff;padding:20px;border:1px solid #ccc;z-index:9999;">
        <h2>Import Contractors</h2>
        <form id="dfs_import_form" enctype="multipart/form-data">
            <p><strong>Option 1: Paste CSV or semicolon-delimited text</strong></p>
            <textarea id="import_data" name="import_data" rows="8" style="width:100%;"></textarea>
            <p><strong>Option 2: Upload a CSV or TXT file</strong></p>
            <input type="file" name="import_file" id="import_file" accept=".csv,.txt"><br><br>
            <button type="submit" class="button button-primary">Import</button>
            <button type="button" id="dfs_cancel_import" class="button">Cancel</button>
        </form>
    </div>
    <script>
        jQuery(document).ready(function ($) {
            $('#dfs_import_contractors').on('click', function () {
                $('#dfs_import_modal').show();
            });
            $('#dfs_cancel_import').on('click', function () {
                $('#dfs_import_modal').hide();
            });
            $('#dfs_import_form').on('submit', function (e) {
                e.preventDefault();
                var formData = new FormData(this);
                formData.append('action', 'dfs_import_contractors');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    contentType: false,
                    processData: false,
                    success: function (resp) {
                        if (resp.success) {
                            alert('Import complete: ' + resp.data.imported + ' records added.');
                            location.reload();
                        } else {
                            alert('Import failed.');
                        }
                    }
                });
            });
        });
    </script>
    <?php
});

// Enhanced import logic
add_action('wp_ajax_dfs_import_contractors', function () {
    global $wpdb;
    $imported = 0;
    $lines = [];

    if (!empty($_FILES['import_file']['tmp_name'])) {
        $content = file_get_contents($_FILES['import_file']['tmp_name']);
        $lines = explode("\n", $content);
    } else
        $table = $wpdb->prefix . 'dfs_contractors';

    foreach ($lines as $line) {
        $line = trim($line);
        if (!$line)
            continue;

        $delimiter = strpos($line, ';') !== false ? ';' : ',';
        $parts = array_map('trim', explode($delimiter, $line));
        if (count($parts) < 8)
            continue;

        list($contractor_name, $contact_name, $address, $city, $state, $zip, $phone, $email) = array_pad($parts, 8, '');
        $opt_out = isset($parts[8]) && strtolower($parts[8]) === 'yes' ? 1 : 0;

        $wpdb->insert($table, [
            'contractor_name' => $contractor_name,
            'contact_name' => $contact_name,
            'address' => $address,
            'city' => $city,
            'state' => $state,
            'zip' => $zip,
            'phone' => $phone,
            'email' => $email,
            'opt_out' => $opt_out,
        ]);
        $imported++;
    }

    wp_send_json_success(['imported' => $imported]);
});

// AJAX handlers for Email Selected
add_action('wp_ajax_dfs_get_templates', 'dfs_get_templates');
function dfs_get_templates()
{
    check_ajax_referer('dfs_email_nonce');
    $templates = get_option('dfs_email_templates', []);
    wp_send_json_success($templates);
}
add_action('wp_ajax_dfs_save_template', 'dfs_save_template');
function dfs_save_template()
{
    check_ajax_referer('dfs_email_nonce');
    $name = sanitize_text_field($_POST['name']);
    $subject = sanitize_text_field($_POST['subject']);
    $body = wp_kses_post($_POST['body']);
    $templates = get_option('dfs_email_templates', []);
    $templates[$name] = ['subject' => $subject, 'body' => $body];
    update_option('dfs_email_templates', $templates);
    wp_send_json_success($templates);
}
add_action('wp_ajax_dfs_delete_template', 'dfs_delete_template');
function dfs_delete_template()
{
    check_ajax_referer('dfs_email_nonce');
    $name = sanitize_text_field($_POST['name']);
    $templates = get_option('dfs_email_templates', []);
    unset($templates[$name]);
    update_option('dfs_email_templates', $templates);
    wp_send_json_success($templates);
}
add_action('wp_ajax_dfs_send_email', 'dfs_send_email');
function dfs_send_email()
{
    check_ajax_referer('dfs_email_nonce');
    $id = intval($_POST['id']);
    $subj = sanitize_text_field($_POST['subject']);
    $body = wp_kses_post($_POST['body']);
    global $wpdb;
    $c = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}dfs_contractors WHERE id=%d", $id));
    if (!$c)
        wp_send_json_error('Not found');
    $body = str_replace(['{contractor_name}', '{city}', '{email}'], [$c->contractor_name, $c->city, $c->email], $body);
    $sent = wp_mail($c->email, $subj, $body);
    if ($sent)
        wp_send_json_success('Sent');
    wp_send_json_error('Failed');
}


/**
 * Render the Dashboard page.
 */
function dfs_render_dashboard()
{
    global $wpdb;
    // Contractor stats
    $table = $wpdb->prefix . 'dfs_contractors';
    $total_contractors = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    $nc_contractors = (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE state = %s", 'NC'));
    $sc_contractors = (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE state = %s", 'SC'));

    // Product stats
    $categories = get_terms(array('taxonomy' => 'product_cat', 'hide_empty' => false));

    echo '<div class="wrap"><h1>Dashboard</h1><div class="dfs-dashboard-cards">';
    echo '<div class="dfs-dashboard-card"><h2>Contractors</h2>';
    echo '<p>Total: ' . esc_html($total_contractors) . '</p>';
    echo '<p>NC: ' . esc_html($nc_contractors) . '</p>';
    echo '<p>SC: ' . esc_html($sc_contractors) . '</p>';
    echo '</div>';
    echo '<div class="dfs-dashboard-card"><h2>Products by Category</h2>';
    echo '<table class="widefat fixed"><thead><tr><th>Category</th><th>Count</th></tr></thead><tbody>';
    if (!empty($categories) && !is_wp_error($categories)) {
        foreach ($categories as $cat) {
            echo '<tr><td>' . esc_html($cat->name) . '</td><td>' . esc_html((int) $cat->count) . '</td></tr>';
        }
    }
    echo '</tbody></table></div>';
    echo '</div></div>';
    echo '<style>
    .dfs-dashboard-cards { display: flex; gap: 20px; margin-top: 20px; }
    .dfs-dashboard-card { background: #fff; padding: 20px; border: 1px solid #e2e2e2; border-radius: 4px; box-shadow: 0 1px 2px rgba(0,0,0,0.1); width: 300px; }
    .dfs-dashboard-card h2 { margin-top: 0; }
    </style>';
}

// Hook DFS Pricelist menus under the Corelentra menu.
add_action('admin_menu', function () {
    add_submenu_page(
        'corelentra-main',
        'Pricelist',
        'Pricelist',
        'manage_options',
        'corelentra-pricelist',
        'dfs_pricelist_module_page'
    );
});


/**
 * Simple Pricelist landing page under Corelentra.
 */
function dfs_pricelist_main_page()
{
    echo '<div class="wrap">';
    echo '<h1>DFS Pricelist</h1>';
    echo '<p>Use the Pricelist Dashboard, Pricelist Settings, View Pricelist, and Pricelist Vendors submenus to manage your catalog and contractors.</p>';
    echo '</div>';
}

/**
 * Render the Pricelist module shell with left-hand navigation.
 */
function dfs_pricelist_module_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    $active_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'dashboard';

    echo '<div class="wrap corelentra-module-wrap corelentra-pricelist-module">';
    echo '<h1>' . esc_html__('DFS Pricelist', 'shipping-quote-control') . '</h1>';

    // Two-column layout using the same module styles.
    echo '<style>
    .corelentra-module-layout{display:flex;gap:20px;align-items:flex-start;}
    .corelentra-module-sidebar{width:20%;max-width:260px;background:#fff;border:1px solid #ccd0d4;border-radius:4px;padding:12px;box-sizing:border-box;}
    .corelentra-module-sidebar h2{font-size:14px;margin:0 0 8px;font-weight:600;}
    .corelentra-module-nav a{display:block;padding:6px 8px;margin:0 0 4px;text-decoration:none;border-radius:3px;}
    .corelentra-module-nav a.active{background:#007cba;color:#fff;}
    .corelentra-module-nav a:not(.active){color:#2271b1;}
    .corelentra-module-content{width:80%;max-width:100%;box-sizing:border-box;}
    @media (max-width:782px){
        .corelentra-module-layout{flex-direction:column;}
        .corelentra-module-sidebar,.corelentra-module-content{width:100%;max-width:none;}
    }
    </style>';

    echo '<div class="corelentra-module-layout">';

    // Sidebar menu.
    echo '<div class="corelentra-module-sidebar">';
    echo '<h2>' . esc_html__('Pricelist Menu', 'shipping-quote-control') . '</h2>';
    echo '<div class="corelentra-module-nav">';
    $base_url = admin_url('admin.php?page=corelentra-pricelist');

    $tabs = array(
        'pricelist' => __('Pricelist', 'shipping-quote-control'),
        'dashboard' => __('Pricelist Dashboard', 'shipping-quote-control'),
        'view' => __('View Pricelist', 'shipping-quote-control'),
        'vendors' => __('Pricelist Vendors', 'shipping-quote-control'),
        'settings' => __('Pricelist Settings', 'shipping-quote-control'),
        'instructions' => __('Pricelist Instructions', 'shipping-quote-control'),
    );

    foreach ($tabs as $tab_key => $label) {
        $url = add_query_arg('tab', $tab_key, $base_url);
        $class = ($active_tab === $tab_key) ? 'active' : '';
        echo '<a class="' . esc_attr($class) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
    }

    echo '</div>'; // .corelentra-module-nav
    echo '</div>'; // .corelentra-module-sidebar

    // Main content.
    echo '<div class="corelentra-module-content">';

    switch ($active_tab) {
        case 'pricelist':
            dfs_pricelist_main_page();
            break;
        case 'view':
            dfs_pricelist_admin_page();
            break;
        case 'vendors':
            dfs_render_fence_contractors();
            break;
        case 'settings':
            dfs_pricelist_settings_page();
            break;
        case 'instructions':
            dfs_pricelist_instructions_page();
            break;
        case 'dashboard':
        default:
            dfs_render_dashboard();
            break;
    }

    echo '</div>'; // .corelentra-module-content
    echo '</div>'; // .corelentra-module-layout
    echo '</div>'; // .wrap
}

/**
 * Instructions page for DFS Pricelist module.
 */
function dfs_pricelist_instructions_page()
{
    echo '<h2>' . esc_html__('How to Use the DFS Pricelist Module', 'shipping-quote-control') . '</h2>';
    echo '<p>' . esc_html__('The DFS Pricelist module lets you generate an HTML and PDF catalog of your WooCommerce products and manage vendor/contractor records.', 'shipping-quote-control') . '</p>';
    echo '<ol>';
    echo '<li>' . esc_html__('Use the Pricelist Dashboard to review summary stats and quick links into the pricelist tools.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Use View Pricelist to preview the HTML catalog version of your products grouped by category.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Use Pricelist Settings to control which categories show, PDF styling, and shortcode behavior.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Use Pricelist Vendors to manage contractor and dealer records that receive your pricelist via email.', 'shipping-quote-control') . '</li>';
    echo '</ol>';
    echo '<p>' . esc_html__('Once configured, you can embed the pricelist on the front-end using the DFS shortcodes and generate downloadable PDFs for customers or vendors.', 'shipping-quote-control') . '</p>';
}
