<?php
/**
 * Scheduler module for Corelentra.
 *
 * Handles appointments for installations, deliveries, on-site quotes/estimates, and invoices.
 * Also provides optional one-way Google Calendar sync when credentials are configured.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Maybe install / upgrade scheduler DB tables.
 */
function corelentra_scheduler_maybe_install() {
    global $wpdb;

    $installed_version = get_option( 'corelentra_scheduler_db_version' );
    $target_version    = '1.0.0';

    if ( $installed_version === $target_version ) {
        return;
    }

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';

    $charset_collate = $wpdb->get_charset_collate();

    $appointments_table = $wpdb->prefix . 'corelentra_appointments';

    $sql_appointments = "CREATE TABLE {$appointments_table} (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        title VARCHAR(255) NOT NULL,
        appointment_type VARCHAR(50) NOT NULL DEFAULT 'installation',
        related_type VARCHAR(50) NOT NULL DEFAULT 'manual',
        related_id BIGINT(20) UNSIGNED NULL,
        customer_name VARCHAR(255) NULL,
        customer_phone VARCHAR(50) NULL,
        customer_email VARCHAR(255) NULL,
        start_datetime DATETIME NOT NULL,
        end_datetime DATETIME NULL,
        all_day TINYINT(1) NOT NULL DEFAULT 0,
        location TEXT NULL,
        status VARCHAR(50) NOT NULL DEFAULT 'scheduled',
        notes LONGTEXT NULL,
        google_event_id VARCHAR(255) NULL,
        created_at DATETIME NOT NULL,
        updated_at DATETIME NOT NULL,
        PRIMARY KEY  (id),
        KEY related_type (related_type),
        KEY related_id (related_id),
        KEY start_datetime (start_datetime),
        KEY status (status)
    ) {$charset_collate};";

    dbDelta( $sql_appointments );

    update_option( 'corelentra_scheduler_db_version', $target_version );
}
add_action( 'admin_init', 'corelentra_scheduler_maybe_install' );

/**
 * Register scheduler settings.
 */
function corelentra_scheduler_register_settings() {
    register_setting(
        'corelentra_scheduler_settings_group',
        'corelentra_scheduler_settings',
        array(
            'type'              => 'array',
            'sanitize_callback' => 'corelentra_scheduler_sanitize_settings',
            'default'           => array(
                'default_duration'    => 60,
                'default_time_step'   => 30,
                'google_client_id'    => '',
                'google_client_secret'=> '',
                'google_calendar_id'  => '',
                'google_access_token' => '',
            ),
        )
    );
}
add_action( 'admin_init', 'corelentra_scheduler_register_settings' );

/**
 * Sanitize scheduler settings.
 */
function corelentra_scheduler_sanitize_settings( $value ) {
    $clean = array();

    $clean['default_duration']  = isset( $value['default_duration'] ) ? max( 15, absint( $value['default_duration'] ) ) : 60;
    $clean['default_time_step'] = isset( $value['default_time_step'] ) ? max( 15, absint( $value['default_time_step'] ) ) : 30;

    $clean['google_client_id']     = isset( $value['google_client_id'] ) ? sanitize_text_field( $value['google_client_id'] ) : '';
    $clean['google_client_secret'] = isset( $value['google_client_secret'] ) ? sanitize_text_field( $value['google_client_secret'] ) : '';
    $clean['google_calendar_id']   = isset( $value['google_calendar_id'] ) ? sanitize_text_field( $value['google_calendar_id'] ) : '';
    $clean['google_access_token']  = isset( $value['google_access_token'] ) ? sanitize_textarea_field( $value['google_access_token'] ) : '';

    return $clean;
}


/**
 * Enqueue assets for Scheduler admin pages (Thickbox for modals).
 */
function corelentra_scheduler_admin_enqueue_assets( $hook ) {
    if ( strpos( $hook, 'corelentra-scheduler' ) === false ) {
        return;
    }
    add_thickbox();
}
add_action( 'admin_enqueue_scripts', 'corelentra_scheduler_admin_enqueue_assets' );

/**
 * Admin menu entries for Scheduler.
 */
function corelentra_scheduler_admin_menu() {
    // Hidden raw scheduler page used for legacy add/edit flows and AJAX, not shown in the left WP menu.
    add_submenu_page(
        'corelentra-main',
        __( 'Scheduler (Raw)', 'shipping-quote-control' ),
        __( 'Scheduler (Raw)', 'shipping-quote-control' ),
        'manage_woocommerce',
        'corelentra-scheduler',
        'corelentra_scheduler_render_main'
    );

    // Public-facing Schedule module entry with internal left-hand navigation.
    add_submenu_page(
        'corelentra-main',
        __( 'Schedule', 'shipping-quote-control' ),
        __( 'Schedule', 'shipping-quote-control' ),
        'manage_woocommerce',
        'corelentra-scheduler-module',
        'corelentra_scheduler_render_module_page'
    );
}
add_action( 'admin_menu', 'corelentra_scheduler_admin_menu' );
/**
 * Hide the raw scheduler submenu so only the unified "Schedule" item shows in the Corelentra menu.
 */
function corelentra_scheduler_hide_raw_submenu() {
    remove_submenu_page( 'corelentra-main', 'corelentra-scheduler' );
}
add_action( 'admin_menu', 'corelentra_scheduler_hide_raw_submenu', 999 );

/**
 * Render the Scheduler module shell with left-hand navigation.
 */
function corelentra_scheduler_render_module_page() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        return;
    }

    $active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'calendar';

    echo '<div class="wrap corelentra-module-wrap corelentra-scheduler-module">';
    echo '<h1>' . esc_html__( 'Scheduler', 'shipping-quote-control' ) . '</h1>';

    // Shared layout styles (reuse the same module flexbox shell).
    echo '<style>
    .corelentra-module-layout{display:flex;gap:20px;align-items:flex-start;}
    .corelentra-module-sidebar{width:20%;max-width:260px;background:#fff;border:1px solid #ccd0d4;border-radius:4px;padding:12px;box-sizing:border-box;}
    .corelentra-module-sidebar h2{font-size:14px;margin:0 0 8px;font-weight:600;}
    .corelentra-module-nav a{display:block;padding:6px 8px;margin:0 0 4px;text-decoration:none;border-radius:3px;}
    .corelentra-module-nav a.active{background:#007cba;color:#fff;}
    .corelentra-module-nav a:not(.active){color:#2271b1;}
    .corelentra-module-content{width:80%;max-width:100%;box-sizing:border-box;}
    @media (max-width:782px){
        .corelentra-module-layout{flex-direction:column;}
        .corelentra-module-sidebar,.corelentra-module-content{width:100%;max-width:none;}
    }
    </style>';

    echo '<div class="corelentra-module-layout">';

    // Sidebar navigation for Scheduler module.
    echo '<div class="corelentra-module-sidebar">';
    echo '<h2>' . esc_html__( 'Scheduler Menu', 'shipping-quote-control' ) . '</h2>';
    echo '<div class="corelentra-module-nav">';
    $base_url = admin_url( 'admin.php?page=corelentra-scheduler-module' );

    $tabs = array(
        'calendar'     => __( 'Scheduler', 'shipping-quote-control' ),
        'settings'     => __( 'Scheduler Settings', 'shipping-quote-control' ),
        'instructions' => __( 'Scheduler Instructions', 'shipping-quote-control' ),
    );

    foreach ( $tabs as $tab_key => $label ) {
        $url   = add_query_arg( 'tab', $tab_key, $base_url );
        $class = ( $active_tab === $tab_key ) ? 'active' : '';
        echo '<a class="' . esc_attr( $class ) . '" href="' . esc_url( $url ) . '">' . esc_html( $label ) . '</a>';
    }

    echo '</div>'; // .corelentra-module-nav
    echo '</div>'; // .corelentra-module-sidebar

    // Main content region.
    echo '<div class="corelentra-module-content">';

    switch ( $active_tab ) {
        case 'settings':
            corelentra_scheduler_render_settings_page();
            break;
        case 'instructions':
            corelentra_scheduler_render_instructions_page();
            break;
        case 'calendar':
        default:
            corelentra_scheduler_render_calendar_page();
            break;
    }

    echo '</div>'; // .corelentra-module-content
    echo '</div>'; // .corelentra-module-layout
    echo '</div>'; // .wrap
}


/**
 * Fetch scheduler settings with defaults applied.
 */
function corelentra_scheduler_get_settings() {
    $defaults = array(
        'default_duration'    => 60,
        'default_time_step'   => 30,
        'google_client_id'    => '',
        'google_client_secret'=> '',
        'google_calendar_id'  => '',
        'google_access_token' => '',
    );

    $settings = get_option( 'corelentra_scheduler_settings', array() );
    if ( ! is_array( $settings ) ) {
        $settings = array();
    }

    return wp_parse_args( $settings, $defaults );
}

/**
 * Render main scheduler page with simple appointment list + add form.
 */

/**
 * AJAX handler to render the scheduler form inside a Thickbox modal.
 */
function corelentra_scheduler_modal_ajax() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this resource.', 'shipping-quote-control' ) );
    }

    corelentra_scheduler_render_main();
    wp_die();
}
add_action( 'wp_ajax_corelentra_scheduler_modal', 'corelentra_scheduler_modal_ajax' );

function corelentra_scheduler_render_main() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        return;
    }

    global $wpdb;
    $appointments_table = $wpdb->prefix . 'corelentra_appointments';

    // Handle create/update.
    if ( isset( $_POST['corelentra_scheduler_nonce'] ) && wp_verify_nonce( $_POST['corelentra_scheduler_nonce'], 'corelentra_scheduler_save_appointment' ) ) {
        $id                 = isset( $_POST['appointment_id'] ) ? absint( $_POST['appointment_id'] ) : 0;
        $title              = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
        $appointment_type   = isset( $_POST['appointment_type'] ) ? sanitize_text_field( wp_unslash( $_POST['appointment_type'] ) ) : 'installation';
        $related_type       = isset( $_POST['related_type'] ) ? sanitize_text_field( wp_unslash( $_POST['related_type'] ) ) : 'manual';
        $related_id         = isset( $_POST['related_id'] ) ? absint( $_POST['related_id'] ) : 0;
        $customer_name      = isset( $_POST['customer_name'] ) ? sanitize_text_field( wp_unslash( $_POST['customer_name'] ) ) : '';
        $customer_phone     = isset( $_POST['customer_phone'] ) ? sanitize_text_field( wp_unslash( $_POST['customer_phone'] ) ) : '';
        $customer_email     = isset( $_POST['customer_email'] ) ? sanitize_email( wp_unslash( $_POST['customer_email'] ) ) : '';
        $start_datetime_raw = isset( $_POST['start_datetime'] ) ? sanitize_text_field( wp_unslash( $_POST['start_datetime'] ) ) : '';
        $end_datetime_raw   = isset( $_POST['end_datetime'] ) ? sanitize_text_field( wp_unslash( $_POST['end_datetime'] ) ) : '';
        $all_day            = isset( $_POST['all_day'] ) ? 1 : 0;
        $location           = isset( $_POST['location'] ) ? sanitize_text_field( wp_unslash( $_POST['location'] ) ) : '';
        $status             = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : 'scheduled';
        $notes              = isset( $_POST['notes'] ) ? wp_kses_post( wp_unslash( $_POST['notes'] ) ) : '';

        $start_datetime = $start_datetime_raw ? date( 'Y-m-d H:i:s', strtotime( $start_datetime_raw ) ) : current_time( 'mysql' );
        $end_datetime   = $end_datetime_raw ? date( 'Y-m-d H:i:s', strtotime( $end_datetime_raw ) ) : null;

        $now = current_time( 'mysql' );

        $data = array(
            'title'           => $title,
            'appointment_type'=> $appointment_type,
            'related_type'    => $related_type,
            'related_id'      => $related_id ?: null,
            'customer_name'   => $customer_name,
            'customer_phone'  => $customer_phone,
            'customer_email'  => $customer_email,
            'start_datetime'  => $start_datetime,
            'end_datetime'    => $end_datetime,
            'all_day'         => $all_day,
            'location'        => $location,
            'status'          => $status,
            'notes'           => $notes,
            'updated_at'      => $now,
        );

        $formats = array(
            '%s','%s','%s','%d','%s','%s','%s','%s','%s','%d','%s','%s','%s','%s'
        );

        if ( $id ) {
            $wpdb->update(
                $appointments_table,
                $data,
                array( 'id' => $id ),
                $formats,
                array( '%d' )
            );

            $appointment_id = $id;
            $action         = 'update';
        } else {
            $data['created_at'] = $now;
            $wpdb->insert( $appointments_table, $data, $formats + array( '%s' ) );
            $appointment_id = (int) $wpdb->insert_id;
            $action         = 'create';
        }

        // Attempt Google sync (non-fatal if it fails).
        corelentra_scheduler_maybe_sync_to_google( $appointment_id, $action );

        // Simple redirect to avoid resubmission.
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            // In modal: close Thickbox and refresh parent calendar/list.
            echo '<script type="text/javascript">(function(){if(window.parent&&typeof window.parent.tb_remove==="function"){window.parent.tb_remove();}if(window.parent&&window.parent.location){window.parent.location.reload();}})();</script>';
            wp_die();
        } else {
            wp_safe_redirect( admin_url( 'admin.php?page=corelentra-scheduler&updated=1' ) );
            exit;
        }
    }

    // Handle delete.
    if ( isset( $_GET['delete'] ) && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'corelentra_scheduler_delete_' . absint( $_GET['delete'] ) ) ) {
        $delete_id = absint( $_GET['delete'] );
        $wpdb->delete( $appointments_table, array( 'id' => $delete_id ), array( '%d' ) );
        corelentra_scheduler_maybe_sync_to_google( $delete_id, 'delete' );

        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            echo '<script type="text/javascript">(function(){if(window.parent&&typeof window.parent.tb_remove==="function"){window.parent.tb_remove();}if(window.parent&&window.parent.location){window.parent.location.reload();}})();</script>';
            wp_die();
        } else {
            wp_safe_redirect( admin_url( 'admin.php?page=corelentra-scheduler&deleted=1' ) );
            exit;
        }
    }

    // Fetch latest 50 appointments.
    $appointments = $wpdb->get_results( "SELECT * FROM {$appointments_table} ORDER BY start_datetime DESC LIMIT 50" );

    $editing = null;
    if ( isset( $_GET['edit'] ) ) {
        $edit_id = absint( $_GET['edit'] );
        $editing = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$appointments_table} WHERE id = %d", $edit_id ) );
    }

    $settings = corelentra_scheduler_get_settings();

    $is_ajax = ( defined( 'DOING_AJAX' ) && DOING_AJAX );

    $editing_id          = $editing ? (int) $editing->id : 0;
    $editing_title       = $editing ? $editing->title : '';
    $editing_type        = $editing ? $editing->appointment_type : 'installation';
    $editing_related_type= $editing ? $editing->related_type : 'manual';
    $editing_related_id  = $editing ? (int) $editing->related_id : 0;
    $editing_customer    = $editing ? $editing->customer_name : '';
    $editing_phone       = $editing ? $editing->customer_phone : '';
    $editing_email       = $editing ? $editing->customer_email : '';
    $editing_start       = $editing ? $editing->start_datetime : current_time( 'mysql' );
    $editing_end         = $editing ? $editing->end_datetime : '';
    $editing_all_day     = $editing ? (int) $editing->all_day : 0;
    $editing_location    = $editing ? $editing->location : '';
    $editing_status      = $editing ? $editing->status : 'scheduled';
    $editing_notes       = $editing ? $editing->notes : '';

    // Are we in a read-only view mode inside a modal?
    $is_view_mode = ( $is_ajax && isset( $_GET['mode'] ) && 'view' === $_GET['mode'] && $editing );

    if ( $is_ajax ) {
        echo '<div class="corelentra-scheduler-modal">';
    } else {
        echo '<div class="wrap corelentra-scheduler-wrap">';
        echo '<h1>' . esc_html__( 'Corelentra Scheduler', 'shipping-quote-control' ) . '</h1>';

        if ( isset( $_GET['updated'] ) ) {
            echo '<div class="updated notice"><p>' . esc_html__( 'Appointment saved.', 'shipping-quote-control' ) . '</p></div>';
        }
        if ( isset( $_GET['deleted'] ) ) {
            echo '<div class="updated notice"><p>' . esc_html__( 'Appointment deleted.', 'shipping-quote-control' ) . '</p></div>';
        }
    }

    // If we are just viewing an existing appointment in a modal, show a read-only summary.
    if ( $is_view_mode ) {
        echo '<h2>' . esc_html__( 'View Appointment', 'shipping-quote-control' ) . '</h2>';

        // Map internal keys to human-readable labels for display.
        $type_labels = array(
            'installation' => __( 'Installation', 'shipping-quote-control' ),
            'quote'        => __( 'Quote Appointment', 'shipping-quote-control' ),
            'wc_delivery'  => __( 'WooCommerce Order Delivery', 'shipping-quote-control' ),
            'phone'        => __( 'Phone Appointment', 'shipping-quote-control' ),
        );
        $related_labels = array(
            'manual'      => __( 'Manual / Standalone', 'shipping-quote-control' ),
            'wc_order'    => __( 'WooCommerce Order', 'shipping-quote-control' ),
            'sales_order' => __( 'Sales Order (Custom)', 'shipping-quote-control' ),
            'crm_record'  => __( 'CRM Record (Future)', 'shipping-quote-control' ),
        );
        $status_labels = array(
            'scheduled' => __( 'Scheduled', 'shipping-quote-control' ),
            'confirmed' => __( 'Confirmed', 'shipping-quote-control' ),
            'completed' => __( 'Completed', 'shipping-quote-control' ),
            'cancelled' => __( 'Cancelled', 'shipping-quote-control' ),
        );

        $display_type    = isset( $type_labels[ $editing_type ] ) ? $type_labels[ $editing_type ] : $editing_type;
        $display_related = isset( $related_labels[ $editing_related_type ] ) ? $related_labels[ $editing_related_type ] : $editing_related_type;
        $display_status  = isset( $status_labels[ $editing_status ] ) ? $status_labels[ $editing_status ] : $editing_status;
        $display_all_day = $editing_all_day ? __( 'Yes (all day event)', 'shipping-quote-control' ) : __( 'No', 'shipping-quote-control' );

        echo '<table class="form-table" role="presentation">';
        echo '<tr><th scope="row">' . esc_html__( 'Title', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_title ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Appointment Type', 'shipping-quote-control' ) . '</th><td>' . esc_html( $display_type ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Related To', 'shipping-quote-control' ) . '</th><td>' . esc_html( $display_related ) . ( $editing_related_id ? ' #' . (int) $editing_related_id : '' ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Customer Name', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_customer ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Customer Phone', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_phone ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Customer Email', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_email ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Start', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_start ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'End', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_end ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'All Day', 'shipping-quote-control' ) . '</th><td>' . esc_html( $display_all_day ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Location', 'shipping-quote-control' ) . '</th><td>' . esc_html( $editing_location ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Status', 'shipping-quote-control' ) . '</th><td>' . esc_html( $display_status ) . '</td></tr>';
        if ( $editing_notes ) {
            echo '<tr><th scope="row">' . esc_html__( 'Internal Notes', 'shipping-quote-control' ) . '</th><td>' . nl2br( esc_html( $editing_notes ) ) . '</td></tr>';
        }
        echo '</table>';
        // If this appointment is tied to a WooCommerce order, show basic order details.
        if ( ( 'wc_order' === $editing_related_type || 'wc_delivery' === $editing_type ) && ! empty( $editing_related_id ) && function_exists( 'wc_get_order' ) ) {
            $order_id = (int) $editing_related_id;
            $order    = wc_get_order( $order_id );

            if ( $order ) {
                // Order header with clickable link.
                $order_edit_url = admin_url( 'post.php?post=' . $order_id . '&action=edit' );
                echo '<h3>' . esc_html__( 'Linked WooCommerce Order', 'shipping-quote-control' ) . '</h3>';
                echo '<p>';
                echo esc_html__( 'Order #', 'shipping-quote-control' ) . (int) $order->get_id();
                echo ' &mdash; <a href="' . esc_url( $order_edit_url ) . '" target="_blank">' . esc_html__( 'Open order in new tab', 'shipping-quote-control' ) . '</a>';
                echo '</p>';

                // Order items table.
                $items = $order->get_items();
                if ( ! empty( $items ) ) {
                    echo '<table class="widefat striped" style="margin-top:10px;">';
                    echo '<thead><tr>';
                    echo '<th>' . esc_html__( 'Product', 'shipping-quote-control' ) . '</th>';
                    echo '<th>' . esc_html__( 'SKU', 'shipping-quote-control' ) . '</th>';
                    echo '<th>' . esc_html__( 'Qty', 'shipping-quote-control' ) . '</th>';
                    echo '<th>' . esc_html__( 'Description', 'shipping-quote-control' ) . '</th>';
                    echo '</tr></thead>';
                    echo '<tbody>';

                    foreach ( $items as $item_id => $item ) {
                        $product      = $item->get_product();
                        $product_id   = $product ? $product->get_id() : 0;
                        $thumb_html   = $product_id ? get_the_post_thumbnail( $product_id, array( 32, 32 ) ) : '';
                        $product_name = $item->get_name();
                        $product_sku  = $product ? $product->get_sku() : '';
                        $qty          = $item->get_quantity();

                        echo '<tr>';
                        echo '<td>';
                        if ( $thumb_html ) {
                            echo '<span style="display:inline-block;vertical-align:middle;margin-right:6px;">' . $thumb_html . '</span>';
                        }
                        echo '<span style="vertical-align:middle;">' . esc_html( $product_name ) . '</span>';
                        echo '</td>';
                        echo '<td>' . esc_html( $product_sku ) . '</td>';
                        echo '<td>' . esc_html( $qty ) . '</td>';
                        echo '<td>' . esc_html( $item->get_name() ) . '</td>';
                        echo '</tr>';
                    }

                    echo '</tbody></table>';
                }
            }
        }



        $view_id = $editing_id;

        $edit_modal_url = admin_url(
            'admin-ajax.php?action=corelentra_scheduler_modal&edit=' . (int) $view_id
        );
        $delete_url = wp_nonce_url(
            admin_url( 'admin-ajax.php?action=corelentra_scheduler_modal&delete=' . (int) $view_id ),
            'corelentra_scheduler_delete_' . (int) $view_id
        );

        echo '<p class="submit">';
        echo '<a href="' . esc_url( $edit_modal_url ) . '" class="button button-primary">' . esc_html__( 'Edit', 'shipping-quote-control' ) . '</a> ';
        echo '<a href="' . esc_url( $delete_url ) . '" class="button button-secondary tb-scheduler-delete-link" data-title="' . esc_attr( $editing_title ) . '" onclick="return tbSchedulerConfirmDelete(this);">' . esc_html__( 'Delete', 'shipping-quote-control' ) . '</a> ';
        echo '<button type="button" class="button" onclick="if(window.parent && window.parent.tb_remove){window.parent.tb_remove();}">' . esc_html__( 'Close', 'shipping-quote-control' ) . '</button>';
        echo '</p>';
        echo '<script type="text/javascript">
function tbSchedulerConfirmDelete(el){
    if(!el){return false;}
    var title = el.getAttribute("data-title") || "";
    var message = "Are you sure you want to delete this appointment";
    if(title){
        message += " \"" + title + "\"";
    }
    message += "?";
    if(window.confirm(message)){
        window.location.href = el.href;
    }
    return false;
}
</script>';

        echo '</div>';
        return;
    }

echo '<h2>' . ( $editing ? esc_html__( 'Edit Appointment', 'shipping-quote-control' ) : esc_html__( 'Add New Appointment', 'shipping-quote-control' ) ) . '</h2>';

    echo '<form method="post" action="">';

    wp_nonce_field( 'corelentra_scheduler_save_appointment', 'corelentra_scheduler_nonce' );
    echo '<input type="hidden" name="appointment_id" value="' . esc_attr( $editing_id ) . '" />';

    echo '<table class="form-table" role="presentation">';

    echo '<tr><th scope="row"><label for="tb_sched_title">' . esc_html__( 'Title', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="title" id="tb_sched_title" type="text" class="regular-text" value="' . esc_attr( $editing_title ) . '" required /></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_type">' . esc_html__( 'Appointment Type', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><select name="appointment_type" id="tb_sched_type">';
    $types = array(
        'installation' => __( 'Installation', 'shipping-quote-control' ),
        'quote'        => __( 'Quote Appointment', 'shipping-quote-control' ),
        'wc_delivery'  => __( 'WooCommerce Order Delivery', 'shipping-quote-control' ),
        'phone'        => __( 'Phone Appointment', 'shipping-quote-control' ),
    );
    foreach ( $types as $key => $label ) {
        echo '<option value="' . esc_attr( $key ) . '" ' . selected( $editing_type, $key, false ) . '>' . esc_html( $label ) . '</option>';
    }
    echo '</select></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_related_type">' . esc_html__( 'Related To', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><select name="related_type" id="tb_sched_related_type">';
    $related_types = array(
        'manual'     => __( 'Manual / Standalone', 'shipping-quote-control' ),
        'wc_order'   => __( 'WooCommerce Order', 'shipping-quote-control' ),
        'sales_order'=> __( 'Sales Order (Custom)', 'shipping-quote-control' ),
        'crm_record' => __( 'CRM Record (Future)', 'shipping-quote-control' ),
    );
    foreach ( $related_types as $key => $label ) {
        echo '<option value="' . esc_attr( $key ) . '" ' . selected( $editing_related_type, $key, false ) . '>' . esc_html( $label ) . '</option>';
    }
    echo '</select> ';
    echo '<input name="related_id" id="tb_sched_related_id" type="number" min="0" value="' . esc_attr( $editing_related_id ) . '" placeholder="' . esc_attr__( 'ID (optional)', 'shipping-quote-control' ) . '" />';
    echo '</td></tr>';
    // WooCommerce Processing orders selector (shown when type is WooCommerce Order Delivery).
    echo '<tr id="tb_sched_wc_order_row" style="display:none;">';
    echo '<th scope="row"><label for="tb_sched_wc_order">' . esc_html__( 'WooCommerce Order (Processing)', 'shipping-quote-control' ) . '</label></th>';
    echo '<td>';
    if ( function_exists( 'wc_get_orders' ) ) {
        $processing_orders = wc_get_orders( array(
            'status' => 'processing',
            'limit'  => 50,
            'orderby'=> 'date',
            'order'  => 'DESC',
        ) );
        echo '<select name="tb_sched_wc_order" id="tb_sched_wc_order">';
        echo '<option value="0">' . esc_html__( 'Select a processing order', 'shipping-quote-control' ) . '</option>';
        foreach ( $processing_orders as $order ) {
            $order_id      = $order->get_id();
            $billing_name  = $order->get_formatted_billing_full_name();
            $billing_phone = $order->get_billing_phone();
            $billing_email = $order->get_billing_email();
            $addr_parts    = array_filter( array(
                $order->get_shipping_address_1() ? $order->get_shipping_address_1() : $order->get_billing_address_1(),
                $order->get_shipping_address_2() ? $order->get_shipping_address_2() : $order->get_billing_address_2(),
                $order->get_shipping_city()      ? $order->get_shipping_city()      : $order->get_billing_city(),
                $order->get_shipping_state()     ? $order->get_shipping_state()     : $order->get_billing_state(),
                $order->get_shipping_postcode()  ? $order->get_shipping_postcode()  : $order->get_billing_postcode(),
            ) );
            $location_str  = implode( ', ', $addr_parts );
            $total_plain   = wp_strip_all_tags( $order->get_formatted_order_total() );
            $order_label   = '#' . $order_id . ' - ' . $billing_name . ' (' . $total_plain . ')';
            $selected      = ( (int) $editing_related_id === (int) $order_id && 'wc_order' === $editing_related_type ) ? ' selected="selected"' : '';
            echo '<option value="' . esc_attr( $order_id ) . '"'
                . $selected
                . ' data-name="' . esc_attr( $billing_name ) . '"'
                . ' data-phone="' . esc_attr( $billing_phone ) . '"'
                . ' data-email="' . esc_attr( $billing_email ) . '"'
                . ' data-location="' . esc_attr( $location_str ) . '">'
                . esc_html( $order_label ) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__( 'These are your most recent WooCommerce orders in Processing status. Selecting one links this appointment to that order and can prefill customer details.', 'shipping-quote-control' ) . '</p>';
    } else {
        echo '<em>' . esc_html__( 'WooCommerce is not active. Install and activate WooCommerce to link deliveries to orders.', 'shipping-quote-control' ) . '</em>';
    }
    echo '</td></tr>';


    echo '<tr><th scope="row"><label for="tb_sched_customer">' . esc_html__( 'Customer Name', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="customer_name" id="tb_sched_customer" type="text" class="regular-text" value="' . esc_attr( $editing_customer ) . '" /></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_phone">' . esc_html__( 'Customer Phone', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="customer_phone" id="tb_sched_phone" type="text" class="regular-text" value="' . esc_attr( $editing_phone ) . '" /></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_email">' . esc_html__( 'Customer Email', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="customer_email" id="tb_sched_email" type="email" class="regular-text" value="' . esc_attr( $editing_email ) . '" /></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_start">' . esc_html__( 'Start', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="start_datetime" id="tb_sched_start" type="datetime-local" value="' . esc_attr( str_replace( ' ', 'T', substr( $editing_start, 0, 16 ) ) ) . '" required /></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_end">' . esc_html__( 'End', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="end_datetime" id="tb_sched_end" type="datetime-local" value="' . esc_attr( $editing_end ? str_replace( ' ', 'T', substr( $editing_end, 0, 16 ) ) : '' ) . '" />';
    echo '<p class="description">' . esc_html__( 'Leave blank to automatically use the default duration from settings.', 'shipping-quote-control' ) . '</p>';
    echo '</td></tr>';

    echo '<tr><th scope="row">' . esc_html__( 'All Day', 'shipping-quote-control' ) . '</th>';
    echo '<td><label><input type="checkbox" name="all_day" value="1" ' . checked( $editing_all_day, 1, false ) . ' /> ' . esc_html__( 'All day event (ignores specific times)', 'shipping-quote-control' ) . '</label></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_location">' . esc_html__( 'Location', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="location" id="tb_sched_location" type="text" class="large-text" size="80" value="' . esc_attr( $editing_location ) . '" /></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_status">' . esc_html__( 'Status', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><select name="status" id="tb_sched_status">';
    $statuses = array(
        'scheduled' => __( 'Scheduled', 'shipping-quote-control' ),
        'confirmed' => __( 'Confirmed', 'shipping-quote-control' ),
        'completed' => __( 'Completed', 'shipping-quote-control' ),
        'cancelled' => __( 'Cancelled', 'shipping-quote-control' ),
    );
    foreach ( $statuses as $key => $label ) {
        echo '<option value="' . esc_attr( $key ) . '" ' . selected( $editing_status, $key, false ) . '>' . esc_html( $label ) . '</option>';
    }
    echo '</select></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_notes">' . esc_html__( 'Internal Notes', 'shipping-quote-control' ) . '</label></th>';
    echo '<td>';
    echo '<textarea name="notes" id="tb_sched_notes" rows="4" class="large-text">' . esc_textarea( $editing_notes ) . '</textarea>';
    echo '<p class="description">' . esc_html__( 'Visible to staff only. Use this for directions, truck/crew notes, gate codes, etc.', 'shipping-quote-control' ) . '</p>';
    echo '</td></tr>';

    echo '</table>';

    submit_button( $editing ? __( 'Update Appointment', 'shipping-quote-control' ) : __( 'Add Appointment', 'shipping-quote-control' ) );

    echo '</form>';

    echo '<hr />';
    echo '<h2>' . esc_html__( 'Upcoming & Recent Appointments', 'shipping-quote-control' ) . '</h2>';

    if ( empty( $appointments ) ) {
        echo '<p>' . esc_html__( 'No appointments found yet.', 'shipping-quote-control' ) . '</p>';
    } else {
        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__( 'ID', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Title', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Type', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Customer', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Start', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'End', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Status', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Related', 'shipping-quote-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Actions', 'shipping-quote-control' ) . '</th>';
        echo '</tr></thead><tbody>';

        foreach ( $appointments as $appt ) {
            $edit_url   = wp_nonce_url( admin_url( 'admin.php?page=corelentra-scheduler&edit=' . (int) $appt->id ), 'corelentra_scheduler_edit_' . (int) $appt->id );
            $delete_url = wp_nonce_url( admin_url( 'admin.php?page=corelentra-scheduler&delete=' . (int) $appt->id ), 'corelentra_scheduler_delete_' . (int) $appt->id );

            echo '<tr>';
            echo '<td>' . (int) $appt->id . '</td>';
            echo '<td>' . esc_html( $appt->title ) . '</td>';
            echo '<td>' . esc_html( $appt->appointment_type ) . '</td>';
            echo '<td>' . esc_html( $appt->customer_name ) . '</td>';
            echo '<td>' . esc_html( $appt->start_datetime ) . '</td>';
            echo '<td>' . esc_html( $appt->end_datetime ) . '</td>';
            echo '<td>' . esc_html( $appt->status ) . '</td>';
            echo '<td>' . esc_html( $appt->related_type ) . ( $appt->related_id ? ' #' . (int) $appt->related_id : '' ) . '</td>';
            echo '<td><a href="' . esc_url( $edit_url ) . '">' . esc_html__( 'Edit', 'shipping-quote-control' ) . '</a> | <a href="' . esc_url( $delete_url ) . '" onclick="return confirm(\'' . esc_js( __( 'Are you sure?', 'shipping-quote-control' ) ) . '\');">' . esc_html__( 'Delete', 'shipping-quote-control' ) . '</a></td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
    }

    // Inline JS for WooCommerce delivery linking and auto-fill.
    echo '<script type="text/javascript">';
    echo '(function(){';
    echo 'var typeSelect=document.getElementById("tb_sched_type");';
    echo 'var wcRow=document.getElementById("tb_sched_wc_order_row");';
    echo 'var wcSelect=document.getElementById("tb_sched_wc_order");';
    echo 'var relatedType=document.getElementById("tb_sched_related_type");';
    echo 'var relatedId=document.getElementById("tb_sched_related_id");';
    echo 'var nameField=document.getElementById("tb_sched_customer");';
    echo 'var phoneField=document.getElementById("tb_sched_phone");';
    echo 'var emailField=document.getElementById("tb_sched_email");';
    echo 'var locField=document.getElementById("tb_sched_location");';
    echo 'function refreshVisibility(){if(!typeSelect||!wcRow){return;}if(typeSelect.value==="wc_delivery"){wcRow.style.display="";if(relatedType){relatedType.value="wc_order";}}else{wcRow.style.display="none";}}';
    echo 'function applyOrderData(){if(!wcSelect||!relatedType||!relatedId){return;}var opt=wcSelect.options[wcSelect.selectedIndex];if(!opt||!opt.value||opt.value==="0"){return;}var oName=opt.getAttribute("data-name")||"";var oPhone=opt.getAttribute("data-phone")||"";var oEmail=opt.getAttribute("data-email")||"";var oLoc=opt.getAttribute("data-location")||"";relatedType.value="wc_order";relatedId.value=opt.value;if(nameField&&!nameField.value){nameField.value=oName;}if(phoneField&&!phoneField.value){phoneField.value=oPhone;}if(emailField&&!emailField.value){emailField.value=oEmail;}if(locField&&!locField.value){locField.value=oLoc;}}';
    echo 'if(typeSelect){typeSelect.addEventListener("change",refreshVisibility);setTimeout(refreshVisibility,50);}';
    echo 'if(wcSelect){wcSelect.addEventListener("change",applyOrderData);setTimeout(applyOrderData,100);}';
    echo '})();';
    echo '</script>';

    echo '</div>';
}

/**
 * Render scheduler settings page.
 */

/**
 * Render the Scheduler Calendar page (simple built‑in calendar, no external libraries).
 */
function corelentra_scheduler_render_calendar_page() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        return;
    }

    global $wpdb;
    $appointments_table = $wpdb->prefix . 'corelentra_appointments';

    $appointments = $wpdb->get_results(
        "SELECT * FROM {$appointments_table} ORDER BY start_datetime ASC"
    );

    $events = array();

    if ( $appointments ) {
        foreach ( $appointments as $appt ) {
            $edit_url = wp_nonce_url(
                admin_url( 'admin.php?page=corelentra-scheduler&appointment_id=' . (int) $appt->id ),
                'corelentra_scheduler_edit_' . (int) $appt->id
            );

            $events[] = array(
                'id'               => (int) $appt->id,
                'title'            => $appt->title,
                'start'            => $appt->start_datetime,
                'end'              => $appt->end_datetime,
                'allDay'           => (bool) $appt->all_day,
                'status'           => $appt->status,
                'appointment_type' => $appt->appointment_type,
                'customer_name'    => $appt->customer_name,
                'editUrl'          => $edit_url,
                'modalEditUrl'     => admin_url( 'admin-ajax.php?action=corelentra_scheduler_modal&edit=' . (int) $appt->id . '&mode=view' ),
            );
        }
    }

    $events_json   = wp_json_encode( $events );
    $scheduler_url = admin_url( 'admin.php?page=corelentra-scheduler' );
    $scheduler_modal_base = admin_url( 'admin-ajax.php?action=corelentra_scheduler_modal' );
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Scheduler Calendar', 'shipping-quote-control' ); ?></h1>
        <p><?php esc_html_e( 'Use the calendar below to view your Corelentra Scheduler appointments. Click a date to create a new appointment, or click an existing appointment to view and edit it.', 'shipping-quote-control' ); ?></p>

        <div class="tb-cal-toolbar">
            <button type="button" class="button button-secondary tb-cal-view" data-view="month"><?php esc_html_e( 'Month', 'shipping-quote-control' ); ?></button>
            <button type="button" class="button button-secondary tb-cal-view" data-view="week"><?php esc_html_e( 'Week', 'shipping-quote-control' ); ?></button>
            <button type="button" class="button button-secondary tb-cal-view" data-view="day"><?php esc_html_e( 'Day', 'shipping-quote-control' ); ?></button>
            <button type="button" class="button button-secondary tb-cal-view" data-view="year"><?php esc_html_e( 'Year', 'shipping-quote-control' ); ?></button>
        </div>

        <div id="corelentra-calendar"></div>
    </div>

    <style>
        #corelentra-calendar {
            margin-top: 15px;
            background: #ffffff;
            padding: 15px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.06);
            min-height: 300px;
        }
        .tb-cal-toolbar {
            margin-top: 10px;
        }
        .tb-cal-toolbar .button.tb-cal-active {
            background: #2271b1;
            color: #ffffff;
        }
        .tb-cal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        .tb-cal-header .tb-cal-title {
            font-weight: 600;
            font-size: 16px;
        }
        .tb-cal-nav button {
            margin-left: 5px;
        }
        .tb-cal-grid {
            width: 100%;
            border-collapse: collapse;
        }
        .tb-cal-grid th,
        .tb-cal-grid td {
            border: 1px solid #e5e5e5;
            vertical-align: top;
            padding: 4px;
            height: 70px;
            position: relative;
        }
        .tb-cal-grid th {
            text-align: center;
            background: #f8f8f8;
            font-weight: 600;
        }
        .tb-cal-day-number {
            font-size: 12px;
            font-weight: 600;
        }
        .tb-cal-day-add {
            position: absolute;
            top: 4px;
            right: 4px;
            cursor: pointer;
            font-weight: 700;
        }
        .tb-cal-event {
            display: block;
            margin-top: 2px;
            font-size: 11px;
            padding: 2px 4px;
            border-radius: 2px;
            background: #e5f1fb;
            cursor: pointer;
            text-overflow: ellipsis;
            white-space: nowrap;
            overflow: hidden;
        }
        .tb-cal-week-row {
            display: flex;
            border-bottom: 1px solid #e5e5e5;
        }
        .tb-cal-week-day {
            flex: 1;
            border-right: 1px solid #e5e5e5;
            padding: 6px;
        }
        .tb-cal-week-day:last-child {
            border-right: none;
        }
        .tb-cal-week-day-title {
            font-weight: 600;
            margin-bottom: 4px;
        }
        .tb-cal-day-view-date {
            font-weight: 600;
            margin-bottom: 6px;
        }
        .tb-cal-year-grid {
            display: flex;
            flex-wrap: wrap;
            gap: 12px;
        }
        .tb-cal-year-month {
            width: calc(25% - 12px);
            border: 1px solid #e5e5e5;
            border-radius: 3px;
            padding: 6px;
        }
        .tb-cal-year-month-title {
            font-weight: 600;
            margin-bottom: 4px;
            font-size: 12px;
        }
        .tb-cal-year-month-count {
            font-size: 11px;
        }
        @media (max-width: 900px) {
            .tb-cal-year-month {
                width: calc(33.333% - 12px);
            }
        }
        @media (max-width: 600px) {
            .tb-cal-year-month {
                width: calc(50% - 12px);
            }
        }
    </style>

    <script type="text/javascript">
    (function() {
        var events      = <?php echo $events_json ? $events_json : '[]'; ?>;
        var schedulerUrl = <?php echo wp_json_encode( $scheduler_url ); ?>;
        var currentDate  = new Date();
        var currentView  = 'month';
        // Open the scheduler in a Thickbox modal when available, otherwise fall back to a normal redirect.
        function openAppointmentModal(targetUrl) {
            if (window.tb_show) {
                var width  = Math.min(window.innerWidth  ? window.innerWidth - 100 : 900, 900);
                var height = Math.min(window.innerHeight ? window.innerHeight - 100 : 800, 800);
                tb_show('', targetUrl + '&TB_iframe=true&width=' + width + '&height=' + height);
            } else {
                window.location.href = targetUrl;
            }
        }


        function toDateOnly(dateStr) {
            if (!dateStr) {
                return null;
            }
            var d = new Date(dateStr.replace(' ', 'T'));
            if (isNaN(d.getTime())) {
                return null;
            }
            return new Date(d.getFullYear(), d.getMonth(), d.getDate());
        }

        function dateKey(dateObj) {
            var y = dateObj.getFullYear();
            var m = ('0' + (dateObj.getMonth() + 1)).slice(-2);
            var d = ('0' + dateObj.getDate()).slice(-2);
            return y + '-' + m + '-' + d;
        }

        // Group events by YYYY-MM-DD key.
        var eventsByDate = {};
        events.forEach(function(ev) {
            var d = toDateOnly(ev.start);
            if (!d) {
                return;
            }
            var key = dateKey(d);
            if (!eventsByDate[key]) {
                eventsByDate[key] = [];
            }
            eventsByDate[key].push(ev);
        });

        function setView(view) {
            currentView = view;
            var buttons = document.querySelectorAll('.tb-cal-view');
            buttons.forEach(function(btn) {
                if (btn.getAttribute('data-view') === view) {
                    btn.classList.add('tb-cal-active');
                } else {
                    btn.classList.remove('tb-cal-active');
                }
            });
            renderCalendar();
        }

        function changePeriod(delta) {
            if (currentView === 'month' || currentView === 'year') {
                currentDate.setMonth(currentDate.getMonth() + delta);
            } else {
                currentDate.setDate(currentDate.getDate() + (delta * 7));
            }
            renderCalendar();
        }

        function goToday() {
            currentDate = new Date();
            renderCalendar();
        }

        function createNav(container, titleText) {
            var header = document.createElement('div');
            header.className = 'tb-cal-header';

            var title = document.createElement('div');
            title.className = 'tb-cal-title';
            title.textContent = titleText;

            var nav = document.createElement('div');
            nav.className = 'tb-cal-nav';

            var prevBtn = document.createElement('button');
            prevBtn.type = 'button';
            prevBtn.className = 'button button-small';
            prevBtn.textContent = '«';
            prevBtn.addEventListener('click', function() { changePeriod(-1); });

            var todayBtn = document.createElement('button');
            todayBtn.type = 'button';
            todayBtn.className = 'button button-small';
            todayBtn.textContent = '<?php echo esc_js( __( 'Today', 'shipping-quote-control' ) ); ?>';
            todayBtn.addEventListener('click', goToday);

            var nextBtn = document.createElement('button');
            nextBtn.type = 'button';
            nextBtn.className = 'button button-small';
            nextBtn.textContent = '»';
            nextBtn.addEventListener('click', function() { changePeriod(1); });

            nav.appendChild(prevBtn);
            nav.appendChild(todayBtn);
            nav.appendChild(nextBtn);

            header.appendChild(title);
            header.appendChild(nav);

            container.appendChild(header);
        }

        function renderMonth(container) {
            var year  = currentDate.getFullYear();
            var month = currentDate.getMonth();
            var firstDay = new Date(year, month, 1);
            var firstWeekday = firstDay.getDay(); // 0 Sunday
            var daysInMonth = new Date(year, month + 1, 0).getDate();

            var monthNames = ['January','February','March','April','May','June','July','August','September','October','November','December'];
            createNav(container, monthNames[month] + ' ' + year);

            var table = document.createElement('table');
            table.className = 'tb-cal-grid';

            var thead = document.createElement('thead');
            var headRow = document.createElement('tr');
            var weekdayNames = ['Sun','Mon','Tue','Wed','Thu','Fri','Sat'];
            for (var i = 0; i < 7; i++) {
                var th = document.createElement('th');
                th.textContent = weekdayNames[i];
                headRow.appendChild(th);
            }
            thead.appendChild(headRow);
            table.appendChild(thead);

            var tbody = document.createElement('tbody');

            var day = 1;
            for (var r = 0; r < 6; r++) {
                var tr = document.createElement('tr');
                for (var c = 0; c < 7; c++) {
                    var td = document.createElement('td');
                    if (r === 0 && c < firstWeekday || day > daysInMonth) {
                        td.innerHTML = '&nbsp;';
                    } else {
                        var cellDate = new Date(year, month, day);
                        var key = dateKey(cellDate);

                        var num = document.createElement('div');
                        num.className = 'tb-cal-day-number';
                        num.textContent = day;
                        td.appendChild(num);

                        var add = document.createElement('span');
                        add.className = 'tb-cal-day-add';
                        add.textContent = '+';
                        add.title = '<?php echo esc_js( __( 'Add appointment', 'shipping-quote-control' ) ); ?>';
                        add.addEventListener('click', function(d) {
                            return function(e) {
                                e.stopPropagation();
                                var url = <?php echo wp_json_encode( $scheduler_modal_base ); ?> + '&start_date=' + encodeURIComponent(dateKey(d));
                                openAppointmentModal(url);
                            };
                        }(cellDate));
                        td.appendChild(add);

                        // Click anywhere in cell to create appointment.
                        td.addEventListener('click', function(d) {
                            return function() {
                                var url = <?php echo wp_json_encode( $scheduler_modal_base ); ?> + '&start_date=' + encodeURIComponent(dateKey(d));
                                openAppointmentModal(url);
                            };
                        }(cellDate));

                        if (eventsByDate[key]) {
                            eventsByDate[key].forEach(function(ev) {
                                var evEl = document.createElement('span');
                                evEl.className = 'tb-cal-event';
                                evEl.textContent = ev.title || '(<?php echo esc_js( __( 'No title', 'shipping-quote-control' ) ); ?>)';
                                evEl.title = ev.customer_name ? ev.customer_name : '';
                                evEl.addEventListener('click', function(e) {
                                    e.stopPropagation();
                                    if (ev.modalEditUrl) {
                                        openAppointmentModal(ev.modalEditUrl);
                                    } else if (ev.editUrl) {
                                        window.location.href = ev.editUrl;
                                    }
                                });
                                td.appendChild(evEl);
                            });
                        }

                        day++;
                    }
                    tr.appendChild(td);
                }
                tbody.appendChild(tr);
                if (day > daysInMonth) {
                    break;
                }
            }

            table.appendChild(tbody);
            container.appendChild(table);
        }

        function renderWeek(container) {
            var start = new Date(currentDate);
            start.setDate(start.getDate() - start.getDay()); // Sunday

            var monthNames = ['January','February','March','April','May','June','July','August','September','October','November','December'];
            var end = new Date(start);
            end.setDate(end.getDate() + 6);
            var title = 'Week of ' + monthNames[start.getMonth()] + ' ' + start.getDate() + ', ' + start.getFullYear();
            createNav(container, title);

            var row = document.createElement('div');
            row.className = 'tb-cal-week-row';

            var weekdayNames = ['Sun','Mon','Tue','Wed','Thu','Fri','Sat'];
            for (var i = 0; i < 7; i++) {
                var cellDate = new Date(start);
                cellDate.setDate(start.getDate() + i);
                var key = dateKey(cellDate);

                var dayDiv = document.createElement('div');
                dayDiv.className = 'tb-cal-week-day';

                var titleDiv = document.createElement('div');
                titleDiv.className = 'tb-cal-week-day-title';
                titleDiv.textContent = weekdayNames[i] + ' ' + (cellDate.getMonth() + 1) + '/' + cellDate.getDate();
                dayDiv.appendChild(titleDiv);

                var add = document.createElement('span');
                add.className = 'tb-cal-day-add';
                add.textContent = '+';
                add.title = '<?php echo esc_js( __( 'Add appointment', 'shipping-quote-control' ) ); ?>';
                add.addEventListener('click', function(d) {
                    return function(e) {
                        e.stopPropagation();
                        var url = <?php echo wp_json_encode( $scheduler_modal_base ); ?> + '&start_date=' + encodeURIComponent(dateKey(d));
                        openAppointmentModal(url);
                    };
                }(cellDate));
                dayDiv.appendChild(add);

                dayDiv.addEventListener('click', function(d) {
                    return function() {
                        var url = <?php echo wp_json_encode( $scheduler_modal_base ); ?> + '&start_date=' + encodeURIComponent(dateKey(d));
                        openAppointmentModal(url);
                    };
                }(cellDate));

                if (eventsByDate[key]) {
                    eventsByDate[key].forEach(function(ev) {
                        var evEl = document.createElement('span');
                        evEl.className = 'tb-cal-event';
                        evEl.textContent = ev.title || '(<?php echo esc_js( __( 'No title', 'shipping-quote-control' ) ); ?>)';
                        evEl.title = ev.customer_name ? ev.customer_name : '';
                        evEl.addEventListener('click', function(e) {
                            e.stopPropagation();
                            if (ev.editUrl) {
                                window.location.href = ev.editUrl;
                            }
                        });
                        dayDiv.appendChild(evEl);
                    });
                }

                row.appendChild(dayDiv);
            }

            container.appendChild(row);
        }

        function renderDay(container) {
            var d = new Date(currentDate);
            var key = dateKey(d);
            var monthNames = ['January','February','March','April','May','June','July','August','September','October','November','December'];
            var title = monthNames[d.getMonth()] + ' ' + d.getDate() + ', ' + d.getFullYear();
            createNav(container, title);

            var dateTitle = document.createElement('div');
            dateTitle.className = 'tb-cal-day-view-date';
            dateTitle.textContent = '<?php echo esc_js( __( 'Appointments for this day', 'shipping-quote-control' ) ); ?>';
            container.appendChild(dateTitle);

            var addBtn = document.createElement('button');
            addBtn.type = 'button';
            addBtn.className = 'button button-secondary';
            addBtn.textContent = '<?php echo esc_js( __( 'Add appointment', 'shipping-quote-control' ) ); ?>';
            addBtn.addEventListener('click', function() {
                var url = schedulerUrl + '&start_date=' + encodeURIComponent(key);
                window.location.href = url;
            });
            container.appendChild(addBtn);

            var list = document.createElement('div');
            list.style.marginTop = '10px';

            if (eventsByDate[key]) {
                eventsByDate[key].forEach(function(ev) {
                    var item = document.createElement('div');
                    item.className = 'tb-cal-event';
                    item.style.display = 'block';
                    item.style.marginBottom = '3px';
                    item.textContent = ev.title || '(<?php echo esc_js( __( 'No title', 'shipping-quote-control' ) ); ?>)';
                    item.addEventListener('click', function() {
                        if (ev.modalEditUrl) {
                            openAppointmentModal(ev.modalEditUrl);
                        } else if (ev.editUrl) {
                            window.location.href = ev.editUrl;
                        }
                    });
                    list.appendChild(item);
                });
            } else {
                var none = document.createElement('div');
                none.textContent = '<?php echo esc_js( __( 'No appointments for this day.', 'shipping-quote-control' ) ); ?>';
                list.appendChild(none);
            }

            container.appendChild(list);
        }

        function renderYear(container) {
            var year = currentDate.getFullYear();
            createNav(container, String(year));

            var yearGrid = document.createElement('div');
            yearGrid.className = 'tb-cal-year-grid';

            var monthNames = ['January','February','March','April','May','June','July','August','September','October','November','December'];

            for (var m = 0; m < 12; m++) {
                var monthDate = new Date(year, m, 1);
                var monthEnd  = new Date(year, m + 1, 0);

                var count = 0;
                for (var d = 1; d <= monthEnd.getDate(); d++) {
                    var tmp = new Date(year, m, d);
                    var key = dateKey(tmp);
                    if (eventsByDate[key]) {
                        count += eventsByDate[key].length;
                    }
                }

                var box = document.createElement('div');
                box.className = 'tb-cal-year-month';

                var title = document.createElement('div');
                title.className = 'tb-cal-year-month-title';
                title.textContent = monthNames[m];
                box.appendChild(title);

                var info = document.createElement('div');
                info.className = 'tb-cal-year-month-count';
                info.textContent = count === 1
                    ? '1 appointment'
                    : count + ' appointments';
                box.appendChild(info);

                box.addEventListener('click', function(mIndex) {
                    return function() {
                        currentDate = new Date(year, mIndex, 1);
                        setView('month');
                    };
                }(m));

                yearGrid.appendChild(box);
            }

            container.appendChild(yearGrid);
        }

        function renderCalendar() {
            var container = document.getElementById('corelentra-calendar');
            if (!container) {
                return;
            }
            container.innerHTML = '';

            if (currentView === 'week') {
                renderWeek(container);
            } else if (currentView === 'day') {
                renderDay(container);
            } else if (currentView === 'year') {
                renderYear(container);
            } else {
                renderMonth(container);
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            var viewButtons = document.querySelectorAll('.tb-cal-view');
            viewButtons.forEach(function(btn) {
                btn.addEventListener('click', function() {
                    var view = btn.getAttribute('data-view');
                    setView(view);
                });
            });

            setView(currentView);
        });
    })();
    
        // Handle delete confirmation inside Thickbox modal.
        document.addEventListener('click', function(e) {
            var link = e.target && e.target.closest ? e.target.closest('.tb-scheduler-delete-link') : null;
            if (!link) {
                return;
            }
            e.preventDefault();
            var title = link.getAttribute('data-title') || '';
            var message = 'Are you sure you want to delete this appointment';
            if (title) {
                message += ' "' + title + '"';
            }
            message += '?';
            if (window.confirm(message)) {
                // Use the href (admin-ajax) so DOING_AJAX path can close the modal and refresh the calendar.
                window.location.href = link.href;
            }
        });
</script>
    <?php
}
function corelentra_scheduler_render_settings_page() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        return;
    }

    $settings = corelentra_scheduler_get_settings();

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'Corelentra Scheduler Settings', 'shipping-quote-control' ) . '</h1>';

    echo '<form method="post" action="options.php">';
    settings_fields( 'corelentra_scheduler_settings_group' );
    echo '<table class="form-table" role="presentation">';

    echo '<tr><th scope="row"><label for="tb_sched_default_duration">' . esc_html__( 'Default Appointment Duration (minutes)', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="corelentra_scheduler_settings[default_duration]" id="tb_sched_default_duration" type="number" min="15" step="15" value="' . esc_attr( $settings['default_duration'] ) . '" />';
    echo '<p class="description">' . esc_html__( 'Used when no explicit end time is provided.', 'shipping-quote-control' ) . '</p></td></tr>';

    echo '<tr><th scope="row"><label for="tb_sched_time_step">' . esc_html__( 'Time Slot Increment (minutes)', 'shipping-quote-control' ) . '</label></th>';
    echo '<td><input name="corelentra_scheduler_settings[default_time_step]" id="tb_sched_time_step" type="number" min="15" step="15" value="' . esc_attr( $settings['default_time_step'] ) . '" />';
    echo '<p class="description">' . esc_html__( 'Controls how your future calendar UI will step through times (15, 30, 60, etc.).', 'shipping-quote-control' ) . '</p></td></tr>';

    // Google Sync Section - Pro Only
    $is_pro = class_exists( 'Corelentra_License' ) && Corelentra_License::is_pro();
    $disabled_attr = $is_pro ? '' : 'disabled="disabled"';
    $pro_msg = $is_pro ? '' : '<br><span style="color:#d63638;font-weight:bold;">' . esc_html__( '(Pro Feature Locked)', 'shipping-quote-control' ) . '</span>';

    echo '<tr><th scope="row">' . esc_html__( 'Google Calendar Client ID', 'shipping-quote-control' ) . $pro_msg . '</th>';
    echo '<td><input name="corelentra_scheduler_settings[google_client_id]" type="text" class="regular-text" value="' . esc_attr( $settings['google_client_id'] ) . '" ' . $disabled_attr . ' />';
    echo '<p class="description">' . esc_html__( 'Optional. From your Google Cloud console OAuth 2.0 credentials.', 'shipping-quote-control' ) . '</p></td></tr>';

    echo '<tr><th scope="row">' . esc_html__( 'Google Calendar Client Secret', 'shipping-quote-control' ) . $pro_msg . '</th>';
    echo '<td><input name="corelentra_scheduler_settings[google_client_secret]" type="text" class="regular-text" value="' . esc_attr( $settings['google_client_secret'] ) . '" ' . $disabled_attr . ' />';
    echo '<p class="description">' . esc_html__( 'Optional. From your Google Cloud console OAuth 2.0 credentials.', 'shipping-quote-control' ) . '</p></td></tr>';

    echo '<tr><th scope="row">' . esc_html__( 'Google Calendar ID', 'shipping-quote-control' ) . $pro_msg . '</th>';
    echo '<td><input name="corelentra_scheduler_settings[google_calendar_id]" type="text" class="regular-text" value="' . esc_attr( $settings['google_calendar_id'] ) . '" ' . $disabled_attr . ' />';
    echo '<p class="description">' . esc_html__( 'The Calendar ID where appointments should be mirrored (for example, your Gmail address).', 'shipping-quote-control' ) . '</p></td></tr>';

    echo '<tr><th scope="row">' . esc_html__( 'Google Access Token (JSON)', 'shipping-quote-control' ) . $pro_msg . '</th>';
    echo '<td><textarea name="corelentra_scheduler_settings[google_access_token]" rows="5" class="large-text" ' . $disabled_attr . '>' . esc_textarea( $settings['google_access_token'] ) . '</textarea>';
    echo '<p class="description">' . esc_html__( 'Advanced: paste a JSON access/refresh token pair if you are comfortable managing this manually. If left empty, Google sync will be skipped.', 'shipping-quote-control' ) . '</p></td></tr>';

    echo '</table>';

    submit_button();

    echo '</form>';
    echo '</div>';
}

/**
 * Render instructions page.
 */
function corelentra_scheduler_render_instructions_page() {
    if ( ! current_user_can( 'manage_woocommerce' ) ) {
        return;
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'Corelentra Scheduler Instructions', 'shipping-quote-control' ) . '</h1>';

    echo '<h2>' . esc_html__( 'Overview', 'shipping-quote-control' ) . '</h2>';
    echo '<p>' . esc_html__( 'The Scheduler module helps you organize installations, deliveries, on-site quotes/estimates, invoice follow-ups, and other field work.', 'shipping-quote-control' ) . '</p>';

    echo '<h2>' . esc_html__( 'Creating Appointments', 'shipping-quote-control' ) . '</h2>';
    echo '<ol>';
    echo '<li>' . esc_html__( 'Go to Corelentra → Scheduler.', 'shipping-quote-control' ) . '</li>';
    echo '<li>' . esc_html__( 'Fill out the appointment title, type, and related WooCommerce order / sales order / CRM record if applicable.', 'shipping-quote-control' ) . '</li>';
    echo '<li>' . esc_html__( 'Enter customer contact info, start/end date-time, and any internal notes for your crews.', 'shipping-quote-control' ) . '</li>';
    echo '<li>' . esc_html__( 'Click “Add Appointment” to save.', 'shipping-quote-control' ) . '</li>';
    echo '</ol>';

    echo '<h2>' . esc_html__( 'Editing and Deleting', 'shipping-quote-control' ) . '</h2>';
    echo '<p>' . esc_html__( 'Use the “Edit” link in the appointment table to update an existing appointment, or “Delete” to remove it. Deleting an appointment will also attempt to remove any mirrored Google Calendar event.', 'shipping-quote-control' ) . '</p>';

    echo '<h2>' . esc_html__( 'Google Calendar Sync (Optional)', 'shipping-quote-control' ) . '</h2>';
    echo '<p>' . esc_html__( 'This version includes a lightweight hook for Google Calendar sync. To enable it you must provide valid API credentials and an access token in the Scheduler Settings.', 'shipping-quote-control' ) . '</p>';
    echo '<p>' . esc_html__( 'If credentials are missing or invalid, your appointments will still be saved inside WordPress and the plugin will simply skip the Google sync step without breaking anything.', 'shipping-quote-control' ) . '</p>';

    echo '<h2>' . esc_html__( 'Linking to Orders and CRM Records', 'shipping-quote-control' ) . '</h2>';
    echo '<p>' . esc_html__( 'For now you can manually enter the related WooCommerce order ID, custom sales order ID, or CRM record ID. Future Corelentra modules can surface these links directly from order or customer screens.', 'shipping-quote-control' ) . '</p>';

    echo '</div>';
}

/**
 * Attempt to sync an appointment to Google Calendar.
 *
 * This is intentionally defensive: if credentials are missing or invalid,
 * the function will fail silently (logged to debug.log if enabled) so that
 * it never breaks the admin experience.
 *
 * @param int    $appointment_id
 * @param string $action create|update|delete
 */

function corelentra_scheduler_maybe_sync_to_google( $appointment_id, $action = 'create' ) {
    // Pro Feature Gate
    if ( ! class_exists( 'Corelentra_License' ) || ! Corelentra_License::is_pro() ) {
        return;
    }

    $settings = corelentra_scheduler_get_settings();

    // 1. Basic check for credentials
    if ( empty( $settings['google_client_id'] ) || empty( $settings['google_client_secret'] ) || empty( $settings['google_calendar_id'] ) ) {
        return;
    }

    global $wpdb;
    $table = $wpdb->prefix . 'corelentra_appointments';
    $appt = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $appointment_id ) );

    if ( ! $appt && $action !== 'delete' ) {
        return;
    }

    // 2. Refresh Token Logic (simplified for brevity, assumes access_token is valid or we have refresh logic elsewhere, 
    // but for now we rely on the user pasting a valid token or a simplified refresh flow).
    // In a production environment, you would check expiration and refresh. 
    $access_token = $settings['google_access_token'];
    
    // If token is JSON (from some exploring tools), try to decode.
    $token_data = json_decode($access_token, true);
    if (json_last_error() === JSON_ERROR_NONE && isset($token_data['access_token'])) {
        $access_token = $token_data['access_token'];
    }

    if ( empty( $access_token ) ) {
        return;
    }

    $calendar_id = $settings['google_calendar_id'];
    $api_url = "https://www.googleapis.com/calendar/v3/calendars/" . urlencode($calendar_id) . "/events";

    // 3. Handle DELETE
    if ( 'delete' === $action ) {
        // We need the google_event_id. If we don't have it (or the row is already gone), we can't delete from Google.
        // For delete action, $appt might be null if we deleted first. 
        // Ideally we should have fetched it before delete. 
        // But the previous placeholder logic called this AFTER delete.
        // So we can't delete from Google if we wiped the DB row unless we passed the google_id.
        // Let's skip delete sync for now unless we change the hook order.
        return; 
    }

    // 4. Handle CREATE / UPDATE
    $start_dt = date('c', strtotime($appt->start_datetime)); // ISO 8601
    $end_dt   = $appt->end_datetime ? date('c', strtotime($appt->end_datetime)) : date('c', strtotime($appt->start_datetime . ' +1 hour'));
    
    // Construct Event Body
    $event_data = array(
        'summary' => $appt->title . ' (' . $appt->customer_name . ')',
        'description' => $appt->notes . "\n\nPhone: " . $appt->customer_phone . "\nType: " . $appt->appointment_type,
        'location' => $appt->location,
    );

    if ( $appt->all_day ) {
        $event_data['start'] = array( 'date' => date('Y-m-d', strtotime($appt->start_datetime)) );
        $event_data['end']   = array( 'date' => date('Y-m-d', strtotime($appt->end_datetime ?: $appt->start_datetime)) );
    } else {
        $event_data['start'] = array( 'dateTime' => $start_dt );
        $event_data['end']   = array( 'dateTime' => $end_dt );
    }
    
    // If we have an existing Google ID, we Update (PUT). Else Insert (POST).
    $google_id = $appt->google_event_id;
    $method = 'POST';
    $endpoint = $api_url;

    if ( ! empty( $google_id ) ) {
        $method = 'PUT';
        $endpoint = $api_url . '/' . $google_id;
    }

    $response = wp_remote_request( $endpoint, array(
        'method' => $method,
        'headers' => array(
            'Authorization' => 'Bearer ' . $access_token,
            'Content-Type'  => 'application/json',
        ),
        'body' => json_encode($event_data),
    ) );

    // 5. Save Google ID if new
    if ( ! is_wp_error( $response ) ) {
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        if ( isset( $data['id'] ) && empty( $google_id ) ) {
            $wpdb->update(
                $table,
                array( 'google_event_id' => $data['id'] ),
                array( 'id' => $appointment_id )
            );
        }
    }
}


/**
 * Shortcode [corelentra_scheduler]
 * Displays a frontend booking form for customers.
 */
function corelentra_scheduler_shortcode() {
    // Process form submission if present
    $message = '';
    if ( isset( $_POST['corelentra_frontend_book_nonce'] ) && wp_verify_nonce( $_POST['corelentra_frontend_book_nonce'], 'corelentra_frontend_book' ) ) {
        global $wpdb;
        $table = $wpdb->prefix . 'corelentra_appointments';

        $name   = sanitize_text_field( $_POST['c_name'] );
        $phone  = sanitize_text_field( $_POST['c_phone'] );
        $email  = sanitize_email( $_POST['c_email'] );
        $date   = sanitize_text_field( $_POST['c_date'] );
        $time   = sanitize_text_field( $_POST['c_time'] );
        $notes  = sanitize_textarea_field( $_POST['c_notes'] );

        // Combine date and time
        $start_dt = date( 'Y-m-d H:i:s', strtotime( "$date $time" ) );
        // Default duration 60 mins from settings or hardcoded
        $settings = corelentra_scheduler_get_settings();
        $duration = !empty($settings['default_duration']) ? (int) $settings['default_duration'] : 60;
        $end_dt   = date( 'Y-m-d H:i:s', strtotime( "$start_dt + $duration minutes" ) );

        $data = array(
            'title'           => $name . ' - Quote Request',
            'appointment_type' => 'quote', // Restricted type
            'related_type'    => 'manual',
            'customer_name'   => $name,
            'customer_phone'  => $phone,
            'customer_email'  => $email,
            'start_datetime'  => $start_dt,
            'end_datetime'    => $end_dt,
            'status'          => 'scheduled',
            'notes'           => $notes . " (Booked via Website)",
            'created_at'      => current_time( 'mysql' ),
            'updated_at'      => current_time( 'mysql' ),
        );

        $wpdb->insert( $table, $data );
        
        // Trigger Google Sync if configured
        $new_id = $wpdb->insert_id;
        corelentra_scheduler_maybe_sync_to_google($new_id, 'create');

        $message = '<div class="corelentra-success-box" style="background:#d4edda;color:#155724;padding:15px;margin-bottom:20px;border-radius:4px;">Request submitted successfully! We will contact you shortly to confirm.</div>';
    }

    ob_start();
    echo '<div class="corelentra-booking-form">';
    echo $message;
    ?>
    <style>
        .corelentra-booking-form { max-width: 600px; margin: 0 auto; font-family: inherit; }
        .corelentra-booking-form label { display: block; margin-bottom: 5px; font-weight: bold; }
        .corelentra-booking-form input, .corelentra-booking-form textarea, .corelentra-booking-form select { width: 100%; padding: 8px; margin-bottom: 15px; border: 1px solid #ddd; border-radius: 4px; }
        .corelentra-booking-form button { background: #007cba; color: #fff; border: none; padding: 10px 20px; border-radius: 4px; cursor: pointer; font-size: 16px; }
        .corelentra-booking-form button:hover { background: #006ba1; }
    </style>
    <form method="post">
        <?php wp_nonce_field( 'corelentra_frontend_book', 'corelentra_frontend_book_nonce' ); ?>
        
        <label>Your Name *</label>
        <input type="text" name="c_name" required>

        <label>Phone Number *</label>
        <input type="text" name="c_phone" required>

        <label>Email Address</label>
        <input type="email" name="c_email">

        <label>Requested Date *</label>
        <input type="date" name="c_date" required min="<?php echo date('Y-m-d'); ?>">

        <label>Requested Time *</label>
        <input type="time" name="c_time" required>

        <label>Notes / Details</label>
        <textarea name="c_notes" rows="4" placeholder="Please describe what you need a quote for..."></textarea>

        <button type="submit">Request Appointment</button>
    </form>
    <?php
    echo '</div>';
    return ob_get_clean();
}
add_shortcode( 'corelentra_scheduler', 'corelentra_scheduler_shortcode' );
