<?php
// Shipping Quote module for Corelentra.
// Loaded conditionally from corelentra.php when enabled in the Modules settings.

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Helper: get the current label for the Shipping Quote option.
 * Falls back to "Request Shipping Quote" if no custom label is set.
 */
function sqc_get_quote_label()
{
    $label = get_option('sqc_label', '');
    if (!$label) {
        $label = __('Request Shipping Quote', 'shipping-quote-control');
    }
    return $label;
}

/**
 * Initialize WooCommerce-related functionality once WooCommerce is loaded.
 */
function sqc_init()
{
    if (!class_exists('WooCommerce')) {
        return;
    }

    // Register shipping method class.
    if (!class_exists('SQC_Shipping_Quote_Method') && class_exists('WC_Shipping_Method')) {

        class SQC_Shipping_Quote_Method extends WC_Shipping_Method
        {

            public function __construct($instance_id = 0)
            {
                $this->id = 'sqc_shipping_quote_method';
                $this->instance_id = absint($instance_id);
                $this->method_title = __('Request Shipping Quote', 'shipping-quote-control');
                $this->method_description = __('Allow customers to request a shipping quote instead of live shipping.', 'shipping-quote-control');
                $this->enabled = 'yes';
                $this->title = sqc_get_quote_label();
                $this->supports = array('shipping-zones', 'instance-settings');

                $this->init();
            }

            public function init()
            {
                $this->init_form_fields();
                $this->init_settings();

                // Ensure the title reflects the global custom label if one is set,
                // but still allow per-instance override if the admin changes it in the zone.
                $global_label = sqc_get_quote_label();
                if (empty($this->settings['title'])) {
                    $this->title = $global_label;
                } else {
                    $this->title = $this->settings['title'];
                }

                add_action(
                    'woocommerce_update_options_shipping_' . $this->id,
                    array($this, 'process_admin_options')
                );
            }

            public function init_form_fields()
            {
                $this->form_fields = array(
                    'enabled' => array(
                        'title' => __('Enable', 'shipping-quote-control'),
                        'type' => 'checkbox',
                        'label' => __('Enable Request Shipping Quote', 'shipping-quote-control'),
                        'default' => 'yes',
                    ),
                    'title' => array(
                        'title' => __('Title', 'shipping-quote-control'),
                        'type' => 'text',
                        'description' => __('This controls the title the customer sees at checkout. If left empty, the global Shipping Quote label from Corelentra → Shipping Quote Settings will be used.', 'shipping-quote-control'),
                        'default' => '',
                    ),
                );
            }

            public function calculate_shipping($package = array())
            {
                $rate = array(
                    'id' => $this->id . ':' . $this->instance_id,
                    'label' => $this->title ? $this->title : sqc_get_quote_label(),
                    'cost' => 0,
                );
                $this->add_rate($rate);
            }
        }
    }

    // Register filters related to WooCommerce.
    add_filter('woocommerce_shipping_methods', 'sqc_add_shipping_method');
    add_filter('woocommerce_package_rates', 'sqc_filter_package_rates', 20, 2);

    // Classic checkout: do not require payment when quote shipping is used.
    add_filter('woocommerce_cart_needs_payment', 'sqc_cart_needs_payment', 20, 2);
    add_filter('woocommerce_order_needs_payment', 'sqc_order_needs_payment', 20, 3);

    // Store API / Blocks: do not require payment when quote shipping is used.
    add_filter('woocommerce_store_api_cart_needs_payment', 'sqc_store_api_cart_needs_payment', 20, 2);

    // After order is created (thank you page), mark as quote if our shipping is used.
    add_action('woocommerce_thankyou', 'sqc_mark_order_as_quote_thankyou', 20, 1);
}

/**
 * Force HTML content type for our custom customer quote email only.
 */
function sqc_mail_content_type_html($content_type)
{
    return 'text/html';
}

/**
 * Disable standard WooCommerce customer on-hold / processing emails for Shipping Quote orders,
 * since we send a custom quote email instead.
 */
function sqc_disable_standard_customer_emails_for_quotes($enabled, $order)
{
    if (!$enabled) {
        return false;
    }
    if (!$order instanceof WC_Order) {
        return $enabled;
    }
    if (sqc_order_uses_quote_shipping($order)) {
        return false;
    }
    return $enabled;
}
add_filter('woocommerce_email_enabled_customer_on_hold_order', 'sqc_disable_standard_customer_emails_for_quotes', 10, 2);
add_filter('woocommerce_email_enabled_customer_processing_order', 'sqc_disable_standard_customer_emails_for_quotes', 10, 2);

add_action('plugins_loaded', 'sqc_init', 20);

/**
 * Add our shipping method.
 */
function sqc_add_shipping_method($methods)
{
    $methods['sqc_shipping_quote_method'] = 'SQC_Shipping_Quote_Method';
    return $methods;
}

/**
 * Helper: determine if current cart is using our quote shipping method.
 */
function sqc_cart_uses_quote_shipping()
{
    if (!function_exists('WC') || !WC()->session) {
        return false;
    }

    $chosen_methods = WC()->session->get('chosen_shipping_methods');
    if (empty($chosen_methods) || !is_array($chosen_methods)) {
        return false;
    }

    foreach ($chosen_methods as $method_id) {
        if (is_string($method_id) && strpos($method_id, 'sqc_shipping_quote_method') === 0) {
            return true;
        }
    }

    return false;
}

/**
 * Helper: determine if an order used our quote shipping method.
 */
function sqc_order_uses_quote_shipping($order)
{
    if (!$order) {
        return false;
    }

    $items = $order->get_items('shipping');
    if (empty($items)) {
        return false;
    }

    foreach ($items as $item) {
        $method_id = $item->get_method_id();
        if (is_string($method_id) && strpos($method_id, 'sqc_shipping_quote_method') === 0) {
            return true;
        }
    }

    return false;
}

/**
 * Filter available shipping rates based on zone methods.
 *
 * - If a zone has deliverable methods (Flat rate, Free shipping, etc.),
 *   we leave rates as-is.
 * - If a zone has only Local Pickup and/or Request Shipping Quote,
 *   we keep only those two in live rates.
 * - If no zone matches and no rates exist, we create fallback Local Pickup
 *   and Request Shipping Quote options so the customer can still continue.
 *
 * Additionally, if we end up with only Request Shipping Quote and no Local Pickup,
 * we will add a synthetic Local Pickup rate so the customer can always choose pickup
 * and pay normally, even if the zone did not have Local Pickup configured.
 *
 * We also normalize all Request Shipping Quote labels to use the global label
 * configured in Corelentra → Shipping Quote Settings.
 */
function sqc_filter_package_rates($rates, $package)
{
    if (!class_exists('WC_Shipping_Zones')) {
        return $rates;
    }

    $label = sqc_get_quote_label();

    $zone = WC_Shipping_Zones::get_zone_matching_package($package);
    if (!$zone) {
        // No zone matched at all: if there are no rates, provide fallback.
        if (empty($rates)) {
            $fallback_rates = array();

            // Local Pickup fallback.
            $fallback_rates['local_pickup_fallback'] = new WC_Shipping_Rate(
                'local_pickup_fallback',
                __('Local pickup', 'shipping-quote-control'),
                0,
                array(),
                'local_pickup'
            );

            // Shipping Quote fallback.
            $fallback_rates['sqc_shipping_quote_method_fallback'] = new WC_Shipping_Rate(
                'sqc_shipping_quote_method_fallback',
                $label,
                0,
                array(),
                'sqc_shipping_quote_method'
            );

            return $fallback_rates;
        }

        // Normalize labels for any existing quote rates.
        foreach ($rates as $rate) {
            if (isset($rate->method_id) && 'sqc_shipping_quote_method' === $rate->method_id) {
                $rate->label = $label;
            }
        }

        return $rates;
    }

    $methods = $zone->get_shipping_methods(true);
    $has_delivery = false;

    foreach ($methods as $method) {
        if (empty($method->id) || 'yes' !== $method->enabled) {
            continue;
        }

        // Anything other than pickup/quote is considered a real delivery method.
        if (!in_array($method->id, array('local_pickup', 'sqc_shipping_quote_method'), true)) {
            $has_delivery = true;
            break;
        }
    }

    if ($has_delivery) {
        // Zone supports real delivery methods; leave rates as-is but normalize quote labels.
        foreach ($rates as $rate) {
            if (isset($rate->method_id) && 'sqc_shipping_quote_method' === $rate->method_id) {
                $rate->label = $label;
            }
        }
        return $rates;
    }

    // No deliverable methods in this zone: keep only Local Pickup + Request Shipping Quote in live rates.
    $filtered = array();
    foreach ($rates as $rate_id => $rate) {
        if (in_array($rate->method_id, array('local_pickup', 'sqc_shipping_quote_method'), true)) {
            if ('sqc_shipping_quote_method' === $rate->method_id) {
                $rate->label = $label;
            }
            $filtered[$rate_id] = $rate;
        }
    }

    // If nothing survived filtering (e.g. zone configured but no matching rate objects),
    // fall back to synthetic Local Pickup + Request Shipping Quote rates.
    if (empty($filtered)) {
        $fallback_rates = array();

        $fallback_rates['local_pickup_fallback'] = new WC_Shipping_Rate(
            'local_pickup_fallback',
            __('Local pickup', 'shipping-quote-control'),
            0,
            array(),
            'local_pickup'
        );

        $fallback_rates['sqc_shipping_quote_method_fallback'] = new WC_Shipping_Rate(
            'sqc_shipping_quote_method_fallback',
            $label,
            0,
            array(),
            'sqc_shipping_quote_method'
        );

        return $fallback_rates;
    }

    // Ensure there is always a Local Pickup option when we are in a "no delivery" zone.
    $has_local_pickup_rate = false;
    foreach ($filtered as $rate) {
        if (isset($rate->method_id) && 'local_pickup' === $rate->method_id) {
            $has_local_pickup_rate = true;
            break;
        }
    }

    if (!$has_local_pickup_rate) {
        $filtered['local_pickup_fallback'] = new WC_Shipping_Rate(
            'local_pickup_fallback',
            __('Local pickup', 'shipping-quote-control'),
            0,
            array(),
            'local_pickup'
        );
    }

    return $filtered;
}

/**
 * Classic checkout: do not require payment when quote shipping is selected.
 */
function sqc_cart_needs_payment($needs_payment, $cart)
{
    if (sqc_cart_uses_quote_shipping()) {
        return false;
    }
    return $needs_payment;
}

function sqc_order_needs_payment($needs_payment, $order, $valid_order_statuses)
{
    if (sqc_order_uses_quote_shipping($order)) {
        return false;
    }
    return $needs_payment;
}

/**
 * Store API / Blocks: do not require payment when quote shipping is selected.
 */
function sqc_store_api_cart_needs_payment($needs_payment, $data)
{
    if (sqc_cart_uses_quote_shipping()) {
        return false;
    }
    return $needs_payment;
}

/**
 * After checkout (thank you page), mark order as quote if needed.
 */
function sqc_mark_order_as_quote_thankyou($order_id)
{
    if (!$order_id) {
        return;
    }

    $order = wc_get_order($order_id);
    if (!$order) {
        return;
    }

    if (!sqc_order_uses_quote_shipping($order)) {
        return;
    }

    // Put order on-hold with a clear note. No payment is taken.
    try {
        $order->update_status('on-hold', __('Customer requested a shipping quote. No payment taken.', 'shipping-quote-control'));
    } catch (Throwable $e) {
        // Fail silently to avoid breaking checkout.
    }

    // Email admin or custom notification email a simple notice.
    try {
        $notify_email = get_option('sqc_notification_email');
        if (!$notify_email) {
            $notify_email = get_option('admin_email');
        }
        if ($notify_email) {
            $subject = sprintf('Shipping Quote Requested - Order #%s', $order->get_order_number());
            $message = "A customer requested a shipping quote for order #" . $order->get_order_number() . ".\n\n";
            $message .= "Log in to WooCommerce, add the shipping cost once you have a quote, and then email the customer a payment link or updated order.\n";

            wp_mail($notify_email, $subject, $message);
        }
    } catch (Throwable $e) {
        // Fail silently; do not interrupt checkout.
    }

    // Optional: send custom HTML email to the customer.
    $enable_customer_email = get_option('sqc_enable_customer_email', 'yes');
    if ('yes' !== $enable_customer_email) {
        return;
    }

    $customer_email = $order->get_billing_email();
    if (!$customer_email) {
        return;
    }

    $template = get_option('sqc_customer_email_template', '');
    // Pro Restriction: Force default template if free version
    if (class_exists('Corelentra_License') && !Corelentra_License::is_pro()) {
        $template = ''; // Forces default below
    }

    if (!$template) {
        $template = '<p>' . __('Thank you for your request.', 'shipping-quote-control') . '</p>';
        $template .= '<p>' . __('We have received your shipping quote request for order {order_number} and your order has been placed on hold pending shipment and payment in full.', 'shipping-quote-control') . '</p>';
        $template .= '<p>' . __('We will review the order, calculate freight, and contact you with final totals and payment options.', 'shipping-quote-control') . '</p>';
    }

    // Build placeholder replacements.
    $billing_name = trim($order->get_formatted_billing_full_name());
    $billing_phone = $order->get_billing_phone();
    $billing_email = $order->get_billing_email();
    $billing_address = $order->get_formatted_billing_address();
    $shipping_address = $order->get_formatted_shipping_address();
    $order_total = function_exists('wc_price') ? strip_tags(wc_price($order->get_total())) : $order->get_total();
    $order_date = function_exists('wc_format_datetime') && $order->get_date_created()
        ? wc_format_datetime($order->get_date_created())
        : $order->get_date_created();

    $replacements = array(
        '{order_number}' => $order->get_order_number(),
        '{order_total}' => $order_total,
        '{order_date}' => $order_date,
        '{customer_name}' => $billing_name,
        '{customer_email}' => $billing_email,
        '{billing_phone}' => $billing_phone,
        '{billing_address}' => $billing_address,
        '{shipping_address}' => $shipping_address,
    );

    $body_html = strtr($template, $replacements);
    $body_html = wpautop($body_html);

    // Optionally append WooCommerce-style order details (similar to core emails, but without product images).
    $include_items = get_option('sqc_include_items_in_email', 'yes');
    if ('yes' === $include_items && function_exists('wc_get_email_order_items')) {
        ob_start();
        ?>
        <h2><?php printf(esc_html__('Order #%s', 'woocommerce'), $order->get_order_number()); ?></h2>
        <table class="td" cellspacing="0" cellpadding="6" style="width:100%; border-collapse:collapse; border:1px solid #eee;"
            border="1">
            <thead>
                <tr>
                    <th scope="col" style="text-align:left;"><?php esc_html_e('Product', 'woocommerce'); ?></th>
                    <th scope="col" style="text-align:center;"><?php esc_html_e('Quantity', 'woocommerce'); ?></th>
                    <th scope="col" style="text-align:right;"><?php esc_html_e('Total', 'woocommerce'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php
                echo wc_get_email_order_items(
                    $order,
                    array(
                        'show_sku' => true,
                        'show_image' => false,
                        'image_size' => array(32, 32),
                        'plain_text' => false,
                        'sent_to_admin' => false,
                    )
                );
                ?>
            </tbody>
            <tfoot>
                <?php
                if (method_exists($order, 'get_order_item_totals')) {
                    $totals = $order->get_order_item_totals();
                    if (!empty($totals)) {
                        foreach ($totals as $total) {
                            ?>
                            <tr>
                                <th scope="row" colspan="2" style="text-align:right;"><?php echo wp_kses_post($total['label']); ?></th>
                                <td style="text-align:right;"><?php echo wp_kses_post($total['value']); ?></td>
                            </tr>
                            <?php
                        }
                    }
                }
                ?>
            </tfoot>
        </table>
        <?php
        $body_html .= ob_get_clean();
    }

    $subject = sprintf(
        __('We received your shipping quote request - Order #%s', 'shipping-quote-control'),
        $order->get_order_number()
    );

    // Send as HTML email without affecting other emails.
    add_filter('wp_mail_content_type', 'sqc_mail_content_type_html');
    wp_mail($customer_email, $subject, $body_html);
    remove_filter('wp_mail_content_type', 'sqc_mail_content_type_html');
}


/**
 * Admin: register submenus & settings for Shipping Quote module.
 */
function sqc_admin_menu()
{
    // Single "Shipping Quote" submenu under Corelentra.
    add_submenu_page(
        'corelentra-main',
        __('Shipping Quote', 'shipping-quote-control'),
        __('Shipping Quote', 'shipping-quote-control'),
        'manage_woocommerce',
        'corelentra-shipping-quote',
        'sqc_render_shipping_quote_module_page'
    );
}
add_action('admin_menu', 'sqc_admin_menu');

/**
 * Register settings for notification email and global label (module-specific).
 */
function sqc_admin_init()
{
    // Notification email for Shipping Quote module (sent to admin).
    register_setting(
        'sqc_settings_group',
        'sqc_notification_email',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => '',
        )
    );

    // Global label for the Shipping Quote option.
    register_setting(
        'sqc_settings_group',
        'sqc_label',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => '',
        )
    );

    // Enable / disable custom customer quote email.
    register_setting(
        'sqc_settings_group',
        'sqc_enable_customer_email',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'yes',
        )
    );

    // Include order items / order details section in the customer email.
    register_setting(
        'sqc_settings_group',
        'sqc_include_items_in_email',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'yes',
        )
    );

    // Customer email template body (HTML, with placeholders).
    register_setting(
        'sqc_settings_group',
        'sqc_customer_email_template',
        array(
            'type' => 'string',
            'sanitize_callback' => 'wp_kses_post',
            'default' => '',
        )
    );

    add_settings_section(
        'sqc_settings_section_main',
        __('Shipping Quote Display & Notifications', 'shipping-quote-control'),
        'sqc_render_settings_section_intro',
        'corelentra-shipping-quote-settings'
    );

    add_settings_field(
        'sqc_label_field',
        __('Shipping Quote Label', 'shipping-quote-control'),
        'sqc_render_label_field',
        'corelentra-shipping-quote-settings',
        'sqc_settings_section_main'
    );

    add_settings_field(
        'sqc_notification_email_field',
        __('Notification Email', 'shipping-quote-control'),
        'sqc_render_notification_email_field',
        'corelentra-shipping-quote-settings',
        'sqc_settings_section_main'
    );

    add_settings_field(
        'sqc_enable_customer_email_field',
        __('Enable Customer Quote Email', 'shipping-quote-control'),
        'sqc_render_enable_customer_email_field',
        'corelentra-shipping-quote-settings',
        'sqc_settings_section_main'
    );

    add_settings_field(
        'sqc_include_items_in_email_field',
        __('Include Order Details in Customer Email', 'shipping-quote-control'),
        'sqc_render_include_items_in_email_field',
        'corelentra-shipping-quote-settings',
        'sqc_settings_section_main'
    );

    add_settings_field(
        'sqc_customer_email_template_field',
        __('Customer Email Content', 'shipping-quote-control'),
        'sqc_render_customer_email_template_field',
        'corelentra-shipping-quote-settings',
        'sqc_settings_section_main'
    );
}

add_action('admin_init', 'sqc_admin_init');

function sqc_render_settings_section_intro()
{
    echo '<p>' . esc_html__('Control how the Shipping Quote option appears at checkout and where notifications are sent when customers request a quote.', 'shipping-quote-control') . '</p>';
}

function sqc_render_label_field()
{
    $value = get_option('sqc_label', '');
    echo '<input type="text" class="regular-text" name="sqc_label" value="' . esc_attr($value) . '" />';
    echo '<p class="description">' . esc_html__('This text is shown to customers when the Shipping Quote option appears at checkout. Leave blank to use the default "Request Shipping Quote".', 'shipping-quote-control') . '</p>';
}

function sqc_render_notification_email_field()
{
    $value = get_option('sqc_notification_email', '');
    echo '<input type="email" class="regular-text" name="sqc_notification_email" value="' . esc_attr($value) . '" />';
    echo '<p class="description">' . esc_html__('Email address that will receive notifications when customers submit a Shipping Quote request. Defaults to the site admin email if left blank.', 'shipping-quote-control') . '</p>';
}

/**
 * Render the Shipping Quote module shell with left-hand navigation.
 */

function sqc_render_enable_customer_email_field()
{
    $value = get_option('sqc_enable_customer_email', 'yes');
    echo '<label><input type="checkbox" name="sqc_enable_customer_email" value="yes" ' . checked($value, 'yes', false) . ' /> ' . esc_html__('Send a custom email to the customer when they request a shipping quote.', 'shipping-quote-control') . '</label>';
}

function sqc_render_include_items_in_email_field()
{
    $value = get_option('sqc_include_items_in_email', 'yes');
    echo '<label><input type="checkbox" name="sqc_include_items_in_email" value="yes" ' . checked($value, 'yes', false) . ' /> ' . esc_html__('Append WooCommerce-style order details to the customer quote email.', 'shipping-quote-control') . '</label>';
}

function sqc_render_customer_email_template_field()
{
    // Pro Restriction Check
    if (class_exists('Corelentra_License') && !Corelentra_License::is_pro()) {
        echo '<div style="background:#f0f0f1;padding:15px;border-left:4px solid #d63638;margin-bottom:15px;">';
        echo '<p><strong>' . esc_html__('Pro Feature Locked', 'shipping-quote-control') . '</strong></p>';
        echo '<p>' . esc_html__('Customizing the email template is available in the Pro version only. The default template will be used.', 'shipping-quote-control') . '</p>';
        echo '</div>';
        return;
    }

    $value = get_option('sqc_customer_email_template', '');
    if (!$value) {
        $value = '<p>' . __('Thank you for your request.', 'shipping-quote-control') . '</p>';
        $value .= '<p>' . __('We have received your shipping quote request for order {order_number} and your order has been placed on hold pending shipment and payment in full.', 'shipping-quote-control') . '</p>';
        $value .= '<p>' . __('We will review the order, calculate freight, and contact you with final totals and payment options.', 'shipping-quote-control') . '</p>';
    }

    if (function_exists('wp_editor')) {
        wp_editor(
            $value,
            'sqc_customer_email_template',
            array(
                'textarea_name' => 'sqc_customer_email_template',
                'textarea_rows' => 10,
            )
        );
    } else {
        echo '<textarea name="sqc_customer_email_template" rows="8" cols="60">' . esc_textarea($value) . '</textarea>';
    }

    echo '<p class="description">';
    echo esc_html__('You can use these placeholders in the email content:', 'shipping-quote-control');
    echo ' {order_number}, {order_total}, {order_date}, {customer_name}, {customer_email}, {billing_phone}, {billing_address}, {shipping_address}.';
    echo '</p>';
}
function sqc_render_shipping_quote_module_page()
{
    if (!current_user_can('manage_woocommerce')) {
        return;
    }

    $active_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'quote';

    echo '<div class="wrap corelentra-module-wrap corelentra-shipping-module">';
    echo '<h1>' . esc_html__('Shipping Quote', 'shipping-quote-control') . '</h1>';

    // Simple two-column layout using flexbox.
    echo '<style>
    .corelentra-module-layout{display:flex;gap:20px;align-items:flex-start;}
    .corelentra-module-sidebar{width:20%;max-width:260px;background:#fff;border:1px solid #ccd0d4;border-radius:4px;padding:12px;box-sizing:border-box;}
    .corelentra-module-sidebar h2{font-size:14px;margin:0 0 8px;font-weight:600;}
    .corelentra-module-nav a{display:block;padding:6px 8px;margin:0 0 4px;text-decoration:none;border-radius:3px;}
    .corelentra-module-nav a.active{background:#007cba;color:#fff;}
    .corelentra-module-nav a:not(.active){color:#2271b1;}
    .corelentra-module-content{width:80%;max-width:100%;box-sizing:border-box;}
    @media (max-width:782px){
        .corelentra-module-layout{flex-direction:column;}
        .corelentra-module-sidebar,.corelentra-module-content{width:100%;max-width:none;}
    }
    </style>';

    echo '<div class="corelentra-module-layout">';

    // Sidebar nav.
    echo '<div class="corelentra-module-sidebar">';
    echo '<h2>' . esc_html__('Shipping Quote Menu', 'shipping-quote-control') . '</h2>';
    echo '<div class="corelentra-module-nav">';
    $base_url = admin_url('admin.php?page=corelentra-shipping-quote');

    $tabs = array(
        'quote' => __('Shipping Quote', 'shipping-quote-control'),
        'settings' => __('Shipping Quote Settings', 'shipping-quote-control'),
        'instructions' => __('Shipping Quote Instructions', 'shipping-quote-control'),
    );

    foreach ($tabs as $tab_key => $label) {
        $url = add_query_arg('tab', $tab_key, $base_url);
        $class = ($active_tab === $tab_key) ? 'active' : '';
        echo '<a class="' . esc_attr($class) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
    }

    echo '</div>'; // .corelentra-module-nav
    echo '</div>'; // .corelentra-module-sidebar

    // Main content.
    echo '<div class="corelentra-module-content">';

    switch ($active_tab) {
        case 'settings':
            sqc_render_shipping_quote_settings_page();
            break;
        case 'instructions':
            sqc_render_shipping_quote_instructions_page();
            break;
        case 'quote':
        default:
            sqc_render_shipping_quote_page();
            break;
    }

    echo '</div>'; // .corelentra-module-content
    echo '</div>'; // .corelentra-module-layout
    echo '</div>'; // .wrap
}

/**
 * Instructions page for Shipping Quote module.
 */
function sqc_render_shipping_quote_instructions_page()
{
    echo '<h2>' . esc_html__('How to Use the Shipping Quote Module', 'shipping-quote-control') . '</h2>';
    echo '<p>' . esc_html__('Use this module to let customers request a manual shipping quote when their ZIP code is not covered by an existing WooCommerce shipping zone.', 'shipping-quote-control') . '</p>';
    echo '<ol>';
    echo '<li>' . esc_html__('Configure your standard WooCommerce shipping zones as usual for areas where you ship automatically.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('In Shipping Quote Settings, set the label, internal notification email, and behavior for quote vs. local pickup.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('When a customer checks out with an unsupported ZIP code, they will see the Shipping Quote option instead of live rates.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('You will receive an email with the order details so you can calculate freight and update the order total.', 'shipping-quote-control') . '</li>';
    echo '</ol>';
    echo '<p>' . esc_html__('Once shipping is finalized, update the order in WooCommerce and move it to Processing when payment is complete.', 'shipping-quote-control') . '</p>';
}

/**
 * Render Shipping Quote overview/instructions page.
 */
function sqc_render_shipping_quote_page()
{
    if (!current_user_can('manage_woocommerce')) {
        return;
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Shipping Quote - How It Works', 'shipping-quote-control') . '</h1>';

    echo '<p><em>' . esc_html__('Tip: See “Shipping Quote Settings” to change the label shown to customers and the notification email address.', 'shipping-quote-control') . '</em></p>';

    echo '<p>' . esc_html__('Corelentra Shipping Quote Control lets you safely capture orders that require manual freight calculation, without forcing customers to guess shipping costs.', 'shipping-quote-control') . '</p>';

    echo '<h2>' . esc_html__('Setup Steps', 'shipping-quote-control') . '</h2>';
    echo '<ol>';
    echo '<li>' . esc_html__('Go to WooCommerce → Settings → Shipping → Shipping Zones.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Edit each zone where you want to allow Shipping Quote and add the Shipping Quote method.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Optionally, add “Local Pickup” to the same zone if you want customers to pick up orders instead of requesting a quote.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('In zones that should be quote-only, remove other deliverable methods and leave only Shipping Quote (and Local Pickup if used).', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Go to Corelentra → Shipping Quote → Shipping Quote Settings to set the label and notification email.', 'shipping-quote-control') . '</li>';
    echo '</ol>';

    echo '<h2>' . esc_html__('Customer Experience', 'shipping-quote-control') . '</h2>';
    echo '<ul>';
    echo '<li>' . esc_html__('Customer adds products to the cart and goes to checkout.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('They choose the Shipping Quote option instead of live rates.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('No payment is required at checkout; the order is placed in an on-hold or pending state.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('You receive an email with the order details and can calculate freight manually.', 'shipping-quote-control') . '</li>';
    echo '</ul>';

    echo '<h2>' . esc_html__('Admin Workflow', 'shipping-quote-control') . '</h2>';
    echo '<ol>';
    echo '<li>' . esc_html__('Open the order from WooCommerce → Orders.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Review the customer’s items, address, and any notes.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Calculate shipping manually or via your freight provider.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Add a shipping line item to the order and update the totals.', 'shipping-quote-control') . '</li>';
    echo '<li>' . esc_html__('Send the customer a payment request or update the order status once payment is received.', 'shipping-quote-control') . '</li>';
    echo '</ol>';

    echo '</div>';
}
function sqc_render_shipping_quote_settings_page()
{
    if (!current_user_can('manage_woocommerce')) {
        return;
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Shipping Quote Settings', 'shipping-quote-control') . '</h1>';
    echo '<form method="post" action="options.php">';
    settings_fields('sqc_settings_group');
    do_settings_sections('corelentra-shipping-quote-settings');
    submit_button();
    echo '</form>';
    echo '</div>';
}