<?php
/**
 * Plugin Name: Corelentra
 * Plugin URI: https://discount-fence-supply.com
 * Description: Corelentra is a complete business operations suite designed to help service, retail, and product-based companies run their workflow directly inside WordPress. It replaces multiple separate plugins with one modern, unified system that manages customers, pricing, shipping quotes, sales, vendors, and more — all from a clean, intuitive dashboard.
 * Author: Kurt Lynch
 * Version: 2.9.0
 * Requires at least: 6.0
 * Tested up to: 6.8.3
 * Requires PHP: 7.4
 * Text Domain: shipping-quote-control
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Helper: get Corelentra modules option with sane defaults.
 *
 * @return array
 */
function corelentra_get_modules()
{
    $modules = get_option('corelentra_modules', array());
    if (!is_array($modules)) {
        $modules = array();
    }

    // Ensure we always have keys for known modules with sensible defaults.
    if (!array_key_exists('shipping_quote', $modules)) {
        $modules['shipping_quote'] = 1;
    }
    if (!array_key_exists('pricelist', $modules)) {
        $modules['pricelist'] = 1;
    }
    if (!array_key_exists('scheduler', $modules)) {
        $modules['scheduler'] = 1;
    }

    return $modules;
}

/**
 * Sanitize modules setting.
 *
 * @param mixed $value
 * @return array
 */
function corelentra_sanitize_modules($value)
{
    $clean = array();

    if (is_array($value)) {
        foreach ($value as $key => $flag) {
            $clean[sanitize_key($key)] = $flag ? 1 : 0;
        }
    }

    // Make sure required keys still exist even if someone unsets them manually.
    $defaults = array(
        'shipping_quote' => 1,
        'pricelist' => 1,
        'scheduler' => 1,
    );

    $clean = wp_parse_args($clean, $defaults);

    return $clean;
}

/**
 * Load enabled modules.
 */
function corelentra_load_modules()
{
    $base = plugin_dir_path(__FILE__);

    // ALways load License Manager
    if (file_exists($base . 'modules/license/license.php')) {
        require_once $base . 'modules/license/license.php';
        Corelentra_License::instance();
    }

    if (!class_exists('WooCommerce')) {
        return;
    }

    $modules = corelentra_get_modules();

    // Load Shipping Quote module if enabled (Safe Loader)
    if (!empty($modules['shipping_quote'])) {
        $module_path = $base . 'modules/shipping-quote/shipping-quote.php';
        if (file_exists($module_path)) {
            // Check for method class existence wrapper inside the file, 
            // OR we rely on the fact that this is hooked to 'plugins_loaded' (prio 5)
            // which might be too early for WC_Shipping_Method if WC loads later.
            // Safer to wrap it in an init hook if the file defines the class directly.
            // But for legacy compat, we verify the file exists.
            require_once $module_path;
        }
    }

    // Load Pricelist module if enabled.
    if (!empty($modules['pricelist'])) {
        $module_path = $base . 'modules/pricelist/dfs-pricelist.php';
        if (file_exists($module_path)) {
            require_once $module_path;
        }
    }

    // Load Scheduler module if enabled.
    if (!empty($modules['scheduler'])) {
        $module_path = $base . 'modules/scheduler/scheduler.php';
        if (file_exists($module_path)) {
            require_once $module_path;
        }
    }
}
add_action('plugins_loaded', 'corelentra_load_modules', 5);

/**
 * Admin: register Corelentra menu and core submenus.
 *
 * Individual modules register their own submenus when loaded.
 */
function corelentra_admin_menu()
{
    // Top-level Corelentra menu, positioned near the top (around 4th position).
    add_menu_page(
        __('Corelentra', 'shipping-quote-control'),
        __('Corelentra', 'shipping-quote-control'),
        'manage_woocommerce',
        'corelentra-main',
        'corelentra_render_dashboard',
        'dashicons-admin-generic',
        4
    );

    // Modules page (module toggles).
    add_submenu_page(
        'corelentra-main',
        __('Module Settings', 'shipping-quote-control'),
        __('Module Settings', 'shipping-quote-control'),
        'manage_woocommerce',
        'corelentra-modules',
        'corelentra_render_modules_page'
    );
}
add_action('admin_menu', 'corelentra_admin_menu');

/**
 * Remove the auto-added duplicate "Corelentra" submenu to keep things clean.
 */
function corelentra_remove_duplicate_submenu()
{
    remove_submenu_page('corelentra-main', 'corelentra-main');
}
add_action('admin_menu', 'corelentra_remove_duplicate_submenu', 999);

/**
 * Core settings for modules registry.
 */
function corelentra_admin_init()
{
    register_setting(
        'corelentra_modules_group',
        'corelentra_modules',
        array(
            'type' => 'array',
            'sanitize_callback' => 'corelentra_sanitize_modules',
            'default' => array(
                'shipping_quote' => 1,
                'pricelist' => 1,
                'scheduler' => 1,
            ),
        )
    );
}
add_action('admin_init', 'corelentra_admin_init');

/**
 * Render Corelentra dashboard page.
 */
function corelentra_render_dashboard()
{
    if (!current_user_can('manage_woocommerce')) {
        return;
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Corelentra Suite', 'shipping-quote-control') . '</h1>';

    echo '<div style="display:flex; gap:20px; margin-top:20px;">';

    // Welcome / Status
    echo '<div style="flex:2; background:#fff; padding:20px; border:1px solid #ccd0d4; box-shadow:0 1px 1px rgba(0,0,0,.04);">';
    echo '<h2>' . esc_html__('Business Operations Suite', 'shipping-quote-control') . '</h2>';
    echo '<p>' . esc_html__('Corelentra unifies your workflow into one system. Manage quotes, vendors, appointments, and catalogs directly from WooCommerce.', 'shipping-quote-control') . '</p>';
    if (class_exists('Corelentra_License') && Corelentra_License::is_pro()) {
        echo '<p style="color:green;font-weight:bold;">' . esc_html__('You are running Corelentra Pro. All features are unlocked.', 'shipping-quote-control') . '</p>';
    } else {
        echo '<p style="color:#d63638;">' . esc_html__('You are running the Free Version. Upgrade to unlock powerful integrations.', 'shipping-quote-control') . '</p>';
        echo '<p><a href="' . admin_url('admin.php?page=corelentra-license') . '" class="button button-primary">' . esc_html__('Activate Pro License', 'shipping-quote-control') . '</a></p>';
    }
    echo '</div>';

    // Feature Breakdown
    echo '<div style="flex:1; background:#fff; padding:20px; border:1px solid #ccd0d4; box-shadow:0 1px 1px rgba(0,0,0,.04);">';
    echo '<h3>' . esc_html__('Feature Status', 'shipping-quote-control') . '</h3>';
    echo '<ul style="list-style:disc; margin-left:20px;">';
    echo '<li><strong>Scheduler:</strong> ' . (class_exists('Corelentra_License') && Corelentra_License::is_pro() ? '✅ Google Sync Active' : '🔒 Google Sync Locked') . '</li>';
    echo '<li><strong>Pricelist:</strong> ' . (class_exists('Corelentra_License') && Corelentra_License::is_pro() ? '✅ PDF & Vendors Active' : '🔒 PDF & Vendors Locked') . '</li>';
    echo '<li><strong>Shipping:</strong> ' . (class_exists('Corelentra_License') && Corelentra_License::is_pro() ? '✅ Custom Emails Active' : '🔒 Custom Emails Locked') . '</li>';
    echo '</ul>';
    echo '</div>';

    echo '</div>'; // End flex container
    echo '</div>'; // End wrap
}

/**
 * Render Modules page (module toggles).
 */
function corelentra_render_modules_page()
{
    if (!current_user_can('manage_woocommerce')) {
        return;
    }

    $modules = corelentra_get_modules();
    $shipping_quote_enabled = !empty($modules['shipping_quote']);
    $pricelist_enabled = !empty($modules['pricelist']);
    $scheduler_enabled = !empty($modules['scheduler']);

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Corelentra Modules', 'shipping-quote-control') . '</h1>';
    echo '<p>' . esc_html__('Activate or deactivate individual Corelentra modules.', 'shipping-quote-control') . '</p>';

    echo '<form method="post" action="options.php">';
    settings_fields('corelentra_modules_group');
    echo '<table class="form-table" role="presentation">';

    // Shipping Quote module toggle.
    echo '<tr>';
    echo '<th scope="row">' . esc_html__('Shipping Quote Module', 'shipping-quote-control') . '</th>';
    echo '<td>';
    echo '<label>';
    echo '<input type="checkbox" name="corelentra_modules[shipping_quote]" value="1" ' . checked($shipping_quote_enabled, true, false) . ' /> ';
    echo esc_html__('Enable the Shipping Quote module', 'shipping-quote-control');
    echo '</label>';
    echo '<p class="description">' . esc_html__('Adds the “Request Shipping Quote” workflow so customers can check out without paying shipping until you calculate freight.', 'shipping-quote-control') . '</p>';
    echo '</td>';
    echo '</tr>';

    // Pricelist module toggle.
    echo '<tr>';
    echo '<th scope="row">' . esc_html__('Pricelist Module', 'shipping-quote-control') . '</th>';
    echo '<td>';
    echo '<label>';
    echo '<input type="checkbox" name="corelentra_modules[pricelist]" value="1" ' . checked($pricelist_enabled, true, false) . ' /> ';
    echo esc_html__('Enable the Pricelist module', 'shipping-quote-control');
    echo '</label>';
    echo '<p class="description">' . esc_html__('Enables the DFS Pricelist tools (dashboard, settings, vendors, and view pricelist) under the Corelentra menu.', 'shipping-quote-control') . '</p>';
    echo '</td>';
    echo '</tr>';

    // Scheduler module toggle.
    echo '<tr>';
    echo '<th scope="row">' . esc_html__('Scheduler Module', 'shipping-quote-control') . '</th>';
    echo '<td>';
    echo '<label>';
    echo '<input type="checkbox" name="corelentra_modules[scheduler]" value="1" ' . checked($scheduler_enabled, true, false) . ' /> ';
    echo esc_html__('Enable the Scheduler module', 'shipping-quote-control');
    echo '</label>';
    echo '<p class="description">' . esc_html__('Provides appointment scheduling for installations, deliveries, on-site quotes/estimates, and more, with optional Google Calendar sync.', 'shipping-quote-control') . '</p>';
    echo '</td>';
    echo '</tr>';

    echo '</table>';

    submit_button();
    echo '</form>';

    echo '</div>';
}


/**
 * One-time migration: rename any existing Trustbridge tables to Corelentra tables.
 * This runs once (or until successful) on plugins_loaded and is guarded by an option flag.
 */
function corelentra_run_table_migration_once()
{
    // Only run in admin or during normal load, and only once.
    if (!function_exists('get_option')) {
        return;
    }

    $already_migrated = get_option('corelentra_tables_migrated', 0);
    if (!empty($already_migrated)) {
        return;
    }

    global $wpdb;

    $old_prefix = $wpdb->prefix . 'trustbridge_';
    $new_prefix = $wpdb->prefix . 'corelentra_';

    // Find all existing Trustbridge tables
    $like = $old_prefix . '%';
    $sql = $wpdb->prepare('SHOW TABLES LIKE %s', $like);
    $old_tables = $wpdb->get_col($sql);

    if (empty($old_tables)) {
        // Nothing to migrate, but mark as done so we don't keep checking.
        update_option('corelentra_tables_migrated', 1);
        return;
    }

    foreach ($old_tables as $old_table_name) {
        // Determine the suffix after "trustbridge_".
        $suffix = substr($old_table_name, strlen($old_prefix));
        if (false === $suffix || '' === $suffix) {
            continue;
        }

        $new_table_name = $new_prefix . $suffix;

        // Only rename if the destination table does not exist.
        $check_sql = $wpdb->prepare('SHOW TABLES LIKE %s', $new_table_name);
        $new_exists = $wpdb->get_var($check_sql);

        if (!$new_exists) {
            // Use RENAME TABLE for an atomic rename operation.
            $rename_sql = "RENAME TABLE `{$old_table_name}` TO `{$new_table_name}`";
            $wpdb->query($rename_sql);
        }
        // If $new_exists is true, skip renaming to avoid overwriting data.
    }

    // Mark as migrated so it does not run again.
    update_option('corelentra_tables_migrated', 1);
}
add_action('plugins_loaded', 'corelentra_run_table_migration_once', 1);
